<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\ChartOfAccount;
use App\Models\JournalEntry;
use Illuminate\Http\Request;

class AccountingController extends Controller
{
    public function index(Request $request)
    {
        $accounts = ChartOfAccount::when($request->search, function ($q, $search) {
            $q->where('name', 'like', "%{$search}%")->orWhere('code', 'like', "%{$search}%");
        })->when($request->type && $request->type !== 'all', function ($q) use ($request) {
            $q->where('type', $request->type);
        })->orderBy('code')->paginate(20);

        return view('admin.accounting.index', compact('accounts'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'code' => 'required|string|unique:chart_of_accounts,code',
            'name' => 'required|string|max:255',
            'type' => 'required|in:ASSET,LIABILITY,EQUITY,REVENUE,EXPENSE',
            'parent_id' => 'nullable|exists:chart_of_accounts,id',
            'normal_balance' => 'required|in:DEBIT,CREDIT',
        ]);

        ChartOfAccount::create($validated);

        return back()->with('success', 'Account created successfully.');
    }

    public function update(Request $request, $id)
    {
        $account = ChartOfAccount::findOrFail($id);
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'type' => 'required|string',
            'is_active' => 'boolean',
        ]);
        $account->update($validated);

        return back()->with('success', 'Account updated.');
    }

    public function destroy($id)
    {
        ChartOfAccount::findOrFail($id)->delete();

        return back()->with('success', 'Account deleted.');
    }

    // Journal Entries
    public function journalIndex(Request $request)
    {
        $entries = JournalEntry::with('creator')
            ->when($request->search, function ($q, $search) {
                $q->where('entry_number', 'like', "%{$search}%")->orWhere('description', 'like', "%{$search}%");
            })->when($request->status && $request->status !== 'all', function ($q) use ($request) {
                $q->where('status', $request->status);
            })->latest('date')->paginate(15);

        return view('admin.accounting.journals', compact('entries'));
    }

    public function journalStore(Request $request)
    {
        $validated = $request->validate([
            'date' => 'required|date',
            'description' => 'required|string',
            'reference' => 'nullable|string',
            'items' => 'required|array|min:2',
            'items.*.account_id' => 'required|exists:chart_of_accounts,id',
            'items.*.description' => 'nullable|string',
            'items.*.debit' => 'nullable|numeric|min:0',
            'items.*.credit' => 'nullable|numeric|min:0',
        ]);

        $totalDebit = collect($validated['items'])->sum('debit');
        $totalCredit = collect($validated['items'])->sum('credit');

        if (abs($totalDebit - $totalCredit) > 0.01) {
            return back()->withErrors(['items' => 'Debits and credits must balance.'])->withInput();
        }

        $entry = JournalEntry::create([
            'entry_number' => 'JE-'.now()->format('Ymd').'-'.str_pad(JournalEntry::whereDate('date', today())->count() + 1, 4, '0', STR_PAD_LEFT),
            'date' => $validated['date'],
            'description' => $validated['description'],
            'reference' => $validated['reference'],
            'total_debit' => $totalDebit,
            'total_credit' => $totalCredit,
            'status' => 'DRAFT',
            'created_by' => auth()->id(),
        ]);

        foreach ($validated['items'] as $item) {
            $entry->items()->create($item);
        }

        return back()->with('success', 'Journal entry created successfully.');
    }

    public function journalApprove($id)
    {
        $entry = JournalEntry::findOrFail($id);
        $this->authorize('approve', $entry);
        $entry->update(['status' => 'APPROVED', 'approved_by' => auth()->id(), 'approved_at' => now()]);

        return back()->with('success', 'Journal entry approved.');
    }
}
