<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Bill;
use App\Models\BillItem;
use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderItem;
use App\Services\AccountingService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class BillController extends Controller
{
    protected $accountingService;

    public function __construct(AccountingService $accountingService)
    {
        $this->accountingService = $accountingService;
    }

    public function index()
    {
        $bills = Bill::with('supplier')->latest()->paginate(10);
        return view('admin.bills.index', compact('bills'));
    }

    public function createFromPurchaseOrder($purchaseOrderId)
    {
        $purchaseOrder = PurchaseOrder::with('items.product', 'supplier')->findOrFail($purchaseOrderId);
        
        // Check if already billed? 
        // For simplicity, we allow multiple bills (partials), but maybe warn.

        return view('admin.bills.create', compact('purchaseOrder'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'supplier_id' => 'required|exists:suppliers,id',
            'bill_number' => 'required|string|unique:bills,bill_number',
            'issue_date' => 'required|date',
            'due_date' => 'required|date',
            'items' => 'required|array',
            'items.*.description' => 'required|string',
            'items.*.total_price' => 'required|numeric',
        ]);

        try {
            DB::beginTransaction();

            $bill = Bill::create([
                'bill_number' => $request->bill_number,
                'supplier_id' => $request->supplier_id,
                'purchase_order_id' => $request->purchase_order_id,
                'issue_date' => $request->issue_date,
                'due_date' => $request->due_date,
                'total_amount' => 0, // Recalculate
                'status' => 'DRAFT',
                'created_by' => auth()->id(),
                'notes' => $request->notes,
            ]);

            $total = 0;

            foreach ($request->items as $itemData) {
                // Skip empty lines if any
                if (empty($itemData['description'])) continue;

                $total += $itemData['total_price'];

                BillItem::create([
                    'bill_id' => $bill->id,
                    'product_id' => $itemData['product_id'] ?? null,
                    'description' => $itemData['description'],
                    'quantity' => $itemData['quantity'] ?? 1,
                    'unit_price' => $itemData['unit_price'] ?? $itemData['total_price'],
                    'total_price' => $itemData['total_price'],
                ]);
            }

            $bill->update(['total_amount' => $total]);

            DB::commit();

            return redirect()->route('admin.bills.index')
                ->with('success', 'تم إنشاء الفاتورة بنجاح. يمكنك الآن مراجعتها وترحيلها.');

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'حدث خطأ: ' . $e->getMessage())->withInput();
        }
    }

    public function show(Bill $bill)
    {
        $bill->load('items', 'supplier');
        return view('admin.bills.show', compact('bill'));
    }

    public function post(Bill $bill)
    {
        if ($bill->status !== 'DRAFT') {
            return back()->with('error', 'الفاتورة مرحلة بالفعل أو ملغاة.');
        }

        try {
            DB::beginTransaction();
            
            $this->accountingService->postBillToLedger($bill);
            
            DB::commit();

            return back()->with('success', 'تم ترحيل الفاتورة وإنشاء القيد المحاسبي بنجاح.');

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'فشل الترحيل: ' . $e->getMessage());
        }
    }
}
