<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\Cms\UpdateSiteSettingsRequest;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;

class CMSController extends Controller
{
    // Site Settings
    public function settingsIndex()
    {
        $settings = \App\Models\SiteSetting::first();

        return view('admin.cms.settings.index', compact('settings'));
    }

    public function settingsUpdate(UpdateSiteSettingsRequest $request)
    {
        $validated = $request->validated();

        $settings = \App\Models\SiteSetting::first() ?? new \App\Models\SiteSetting;
        $settings->fill($validated);
        $settings->save();

        $this->clearCmsCaches();

        return back()->with('success', 'تم تحديث الإعدادات بنجاح.');
    }

    // Sliders
    public function slidersIndex()
    {
        $sliders = \App\Models\Slider::orderBy('order')->get();

        return view('admin.cms.sliders.index', compact('sliders'));
    }

    public function sliderStore(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string',
            'image_url' => 'required|url',
            'order' => 'integer',
        ]);

        \App\Models\Slider::create($validated);
        $this->clearCmsCaches();

        return back()->with('success', 'Slider created successfully.');
    }

    public function sliderDestroy($id)
    {
        \App\Models\Slider::destroy($id);
        $this->clearCmsCaches();

        return back()->with('success', 'Slider deleted successfully.');
    }

    // Knowledge Base
    public function kbIndex()
    {
        $categories = \App\Models\KnowledgeBaseCategory::withCount('articles')->get();

        return view('admin.cms.kb.index', compact('categories'));
    }

    public function kbCategoryStore(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|unique:knowledge_base_categories,name',
            'slug' => 'required|string|unique:knowledge_base_categories,slug',
        ]);

        \App\Models\KnowledgeBaseCategory::create($validated);

        return back()->with('success', 'Category created successfully.');
    }

    // Company Stats
    public function statsIndex()
    {
        $stats = \App\Models\CompanyStat::orderBy('order')->get();

        return view('admin.cms.stats.index', compact('stats'));
    }

    public function statsStore(Request $request)
    {
        $validated = $request->validate([
            'label' => 'required|string',
            'value' => 'required|string',
            'icon' => 'nullable|string',
            'order' => 'integer',
        ]);
        \App\Models\CompanyStat::create($validated);
        $this->clearCmsCaches();

        return back()->with('success', 'Stat created successfully.');
    }

    public function statsDestroy($id)
    {
        \App\Models\CompanyStat::destroy($id);
        $this->clearCmsCaches();

        return back()->with('success', 'Stat deleted successfully.');
    }

    // Company Values
    public function valuesIndex()
    {
        $values = \App\Models\CompanyValue::orderBy('order')->get();

        return view('admin.cms.values.index', compact('values'));
    }

    public function valuesStore(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string',
            'description' => 'required|string',
            'icon' => 'nullable|string',
            'order' => 'integer',
        ]);
        \App\Models\CompanyValue::create($validated);
        $this->clearCmsCaches();

        return back()->with('success', 'Value created successfully.');
    }

    public function valuesDestroy($id)
    {
        \App\Models\CompanyValue::destroy($id);
        $this->clearCmsCaches();

        return back()->with('success', 'Value deleted successfully.');
    }

    // Timeline Events
    public function timelineIndex()
    {
        $events = \App\Models\TimelineEvent::orderBy('event_date')->get();

        return view('admin.cms.timeline.index', compact('events'));
    }

    public function timelineStore(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string',
            'event_date' => 'required|date',
            'description' => 'nullable|string',
            'icon' => 'nullable|string',
            'order' => 'integer',
        ]);
        \App\Models\TimelineEvent::create($validated);
        $this->clearCmsCaches();

        return back()->with('success', 'Timeline event created successfully.');
    }

    public function timelineDestroy($id)
    {
        \App\Models\TimelineEvent::destroy($id);
        $this->clearCmsCaches();

        return back()->with('success', 'Timeline event deleted successfully.');
    }

    // Service Items
    public function serviceItemsIndex()
    {
        $items = \App\Models\ServiceItem::orderBy('order')->get();

        return view('admin.cms.service-items.index', compact('items'));
    }

    public function serviceItemsStore(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string',
            'category' => 'nullable|string',
            'price' => 'nullable|numeric',
            'description' => 'nullable|string',
            'icon' => 'nullable|string',
            'order' => 'integer',
        ]);
        \App\Models\ServiceItem::create($validated);
        $this->clearCmsCaches();

        return back()->with('success', 'Service Item created successfully.');
    }

    public function serviceItemsDestroy($id)
    {
        \App\Models\ServiceItem::destroy($id);
        $this->clearCmsCaches();

        return back()->with('success', 'Service Item deleted successfully.');
    }

    // Company Features
    public function featuresIndex()
    {
        $features = \App\Models\CompanyFeature::orderBy('order')->get();

        return view('admin.cms.features.index', compact('features'));
    }

    public function featuresStore(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string',
            'description' => 'nullable|string',
            'icon' => 'nullable|string',
            'order' => 'integer',
        ]);
        \App\Models\CompanyFeature::create($validated);
        $this->clearCmsCaches();

        return back()->with('success', 'Feature created successfully.');
    }

    public function featuresDestroy($id)
    {
        \App\Models\CompanyFeature::destroy($id);
        $this->clearCmsCaches();

        return back()->with('success', 'Feature deleted successfully.');
    }

    // Contact Info
    public function contactIndex()
    {
        $contact = \App\Models\ContactInfo::first();

        return view('admin.cms.contact.index', compact('contact'));
    }

    public function contactUpdate(Request $request)
    {
        $validated = $request->validate([
            'phone' => 'nullable|string',
            'email' => 'nullable|email',
            'address' => 'nullable|string',
            'map_url' => 'nullable|string',
        ]);

        $contact = \App\Models\ContactInfo::first() ?? new \App\Models\ContactInfo;
        $contact->fill($validated);
        $contact->save();
        $this->clearCmsCaches();

        return back()->with('success', 'Contact Info updated successfully.');
    }

    private function clearCmsCaches(): void
    {
        Cache::forget('site_settings');
        Cache::forget('footer_data');
    }
}
