<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

class CrmController extends Controller
{
    // Customers
    public function customersIndex(Request $request)
    {
        $query = \App\Models\CustomerProfile::with('user');

        if ($request->filled('search')) {
            $search = $request->search;
            $query->whereHas('user', function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                    ->orWhere('email', 'like', "%{$search}%")
                    ->orWhere('phone', 'like', "%{$search}%");
            });
        }

        $customers = $query->latest()->paginate(10);

        return view('admin.crm.customers.index', compact('customers'));
    }

    public function customerShow($id)
    {
        $customer = \App\Models\CustomerProfile::with(['user', 'leads', 'opportunities'])->findOrFail($id);

        return view('admin.crm.customers.show', compact('customer'));
    }

    // Leads
    public function leadsIndex(Request $request)
    {
        $query = \App\Models\Lead::with(['customerProfile.user', 'assignedTo']);

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where('first_name', 'like', "%{$search}%")
                ->orWhere('last_name', 'like', "%{$search}%")
                ->orWhere('email', 'like', "%{$search}%")
                ->orWhere('company', 'like', "%{$search}%");
        }

        if ($request->filled('status') && $request->status !== 'all') {
            $query->where('status', $request->status);
        }

        $leads = $query->latest()->paginate(10);

        return view('admin.crm.leads.index', compact('leads'));
    }

    public function leadStore(Request $request)
    {
        $validated = $request->validate([
            'first_name' => 'required|string',
            'last_name' => 'nullable|string',
            'email' => 'nullable|email',
            'phone' => 'nullable|string',
            'company' => 'nullable|string',
            'customer_id' => 'nullable|exists:customer_profiles,id',
            'status' => 'required|string',
            'source' => 'required|string',
        ]);

        $lead = \App\Models\Lead::create(array_merge($validated, [
            'lead_number' => 'LD-'.now()->format('Ymd').'-'.strtoupper(\Illuminate\Support\Str::random(6)),
            'assigned_to' => auth()->id(),
        ]));

        return back()->with('success', 'Lead created successfully.');
    }

    // Opportunities
    public function opportunitiesIndex(Request $request)
    {
        $query = \App\Models\Opportunity::with(['customerProfile.user', 'vehicle']);

        if ($request->filled('status') && $request->status !== 'all') {
            $query->where('stage', $request->status);
        }

        $opportunities = $query->latest()->paginate(10);

        return view('admin.crm.opportunities.index', compact('opportunities'));
    }

    public function opportunityStore(Request $request)
    {
        $validated = $request->validate([
            'customer_id' => 'required|exists:customer_profiles,id',
            'title' => 'required|string',
            'stage' => 'required|string',
            'value' => 'nullable|numeric|min:0',
            'expected_close_date' => 'nullable|date',
        ]);

        \App\Models\Opportunity::create($validated);

        return back()->with('success', 'Opportunity created successfully.');
    }
}
