<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\Finance\StorePaymentRequest;
use App\Jobs\GenerateInvoicePdfJob;
use App\Models\User;
use App\Services\FinanceService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class FinanceController extends Controller
{
    public function __construct(
        private FinanceService $financeService
    ) {}

    public function index(Request $request)
    {
        $query = \App\Models\Invoice::with(['customer', 'branch', 'items']);

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('invoice_number', 'like', "%{$search}%")
                    ->orWhereHas('customer', function ($customerQuery) use ($search) {
                        $customerQuery->where('name', 'like', "%{$search}%");
                    });
            });
        }

        if ($request->filled('status') && $request->status !== 'all') {
            $query->where('status', $request->status);
        }

        $invoices = $query->latest('issue_date')->paginate(10);

        return view('admin.finance.invoices.index', compact('invoices'));
    }

    public function show($id)
    {
        $invoice = \App\Models\Invoice::with(['customer', 'branch', 'items', 'payments'])->findOrFail($id);

        return view('admin.finance.invoices.show', compact('invoice'));
    }

    public function create()
    {
        $customers = User::query()
            ->where('role', 'CUSTOMER')
            ->orderBy('name')
            ->select('id', 'name', 'email')
            ->get();

        return view('admin.finance.invoices.create', compact('customers'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'customer_id' => 'required|exists:users,id',
            'branch_id' => 'nullable|exists:branches,id',
            'vehicle_id' => 'nullable|exists:vehicles,id',
            'type' => 'required|string|in:SALE,SALES,SERVICE,PARTS,RENTAL,OTHER',
            'issue_date' => 'required|date',
            'due_date' => 'required|date|after_or_equal:issue_date',
            'items' => 'required|array|min:1',
            'items.*.description' => 'required|string|max:255',
            'items.*.quantity' => 'required|integer|min:1',
            'items.*.unit_price' => 'required|numeric|min:0',
            'items.*.tax_rate' => 'nullable|numeric|min:0|max:100',
            'notes' => 'nullable|string|max:2000',
            'terms' => 'nullable|string|max:2000',
        ]);

        if ($validated['type'] === 'SALES') {
            $validated['type'] = 'SALE';
        }

        $invoice = $this->financeService->createInvoice($validated, $validated['items']);
        GenerateInvoicePdfJob::dispatch((string) $invoice->id);

        return redirect()->route('admin.finance.index')->with('success', 'Invoice created successfully.');
    }

    public function paymentsIndex(Request $request)
    {
        $query = \App\Models\Payment::with(['customer', 'branch', 'invoice']);

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('transaction_id', 'like', "%{$search}%")
                    ->orWhereHas('customer', function ($customerQuery) use ($search) {
                        $customerQuery->where('name', 'like', "%{$search}%");
                    });
            });
        }

        $payments = $query->latest()->paginate(10);

        return view('admin.finance.payments.index', compact('payments'));
    }

    public function storePayment(StorePaymentRequest $request)
    {
        $validated = $request->validated();

        $this->financeService->recordPayment(array_merge($validated, [
            'branch_id' => auth()->user()->branch_id,
        ]));

        return back()->with('success', 'Payment recorded successfully.');
    }

    public function exportPdf($id)
    {
        $invoice = \App\Models\Invoice::with(['customer', 'items', 'vehicle', 'payments'])->findOrFail($id);
        $pdfPath = GenerateInvoicePdfJob::pathFor((string) $invoice->id);

        if (! Storage::disk('local')->exists($pdfPath)) {
            $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadView('pdf.invoice', compact('invoice'));
            $pdf->setPaper('a4');
            Storage::disk('local')->put($pdfPath, $pdf->output());
        }

        return Storage::disk('local')->download($pdfPath, "invoice-{$invoice->invoice_number}.pdf");
    }
}
