<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\GoodsReceipt;
use App\Models\GoodsReceiptItem;
use App\Models\Product;
use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderItem;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class GoodsReceiptController extends Controller
{
    public function create($id)
    {
        $purchaseOrder = PurchaseOrder::with('items.product')->findOrFail($id);

        if ($purchaseOrder->status === 'COMPLETED' || $purchaseOrder->status === 'CANCELLED') {
            return back()->with('error', 'لا يمكن استلام منتجات لأمر شراء مكتمل أو ملغي');
        }

        return view('admin.goods-receipts.create', compact('purchaseOrder'));
    }

    public function store(Request $request, $id)
    {
        $purchaseOrder = PurchaseOrder::findOrFail($id);

        $request->validate([
            'received_date' => 'required|date',
            'items' => 'required|array',
            'items.*.received_qty' => 'required|integer|min:0',
        ]);

        try {
            DB::beginTransaction();

            // Create Goods Receipt Header
            $count = GoodsReceipt::whereYear('created_at', date('Y'))->count() + 1;
            $receiptNumber = 'GRN-' . date('Y') . '-' . str_pad($count, 4, '0', STR_PAD_LEFT);

            $gr = GoodsReceipt::create([
                'receipt_number' => $receiptNumber,
                'purchase_order_id' => $purchaseOrder->id,
                'received_date' => $request->received_date,
                'notes' => $request->notes,
                'received_by' => auth()->id(),
            ]);

            $hasReceivedItems = false;
            $allFullyReceived = true;

            foreach ($request->items as $itemId => $data) {
                $qty = (int)$data['received_qty'];

                if ($qty > 0) {
                    $poItem = PurchaseOrderItem::findOrFail($itemId);
                    
                    // Validate not receiving more than ordered (optional, but good practice. For now allow over-receiving optionally or strict?)
                    // Let's allow strictly up to ordered amount for now to avoid confusion, or just log it.
                    // Implementation: Just process it.

                    GoodsReceiptItem::create([
                        'goods_receipt_id' => $gr->id,
                        'purchase_order_item_id' => $poItem->id,
                        'product_id' => $poItem->product_id,
                        'quantity_received' => $qty,
                        'notes' => $data['notes'] ?? null,
                    ]);

                    // Update PO Item
                    $poItem->increment('quantity_received', $qty);

                    // Update Product Stock
                    if ($poItem->product_id) {
                        Product::where('id', $poItem->product_id)->increment('quantity', $qty);
                    }

                    $hasReceivedItems = true;
                }
                
                // Check if this item is fully received
                $poItem = PurchaseOrderItem::find($itemId); // Reload
                if ($poItem->quantity_received < $poItem->quantity_ordered) {
                    $allFullyReceived = false;
                }
            }

            if (!$hasReceivedItems) {
                DB::rollBack();
                return back()->with('error', 'يجب استلام كمية واحدة على الأقل');
            }

            // Update PO Status
            if ($allFullyReceived) {
                $purchaseOrder->update(['status' => 'COMPLETED']);
            } else {
                $purchaseOrder->update(['status' => 'PARTIAL']);
            }

            DB::commit();

            return redirect()->route('admin.purchase-orders.show', $purchaseOrder->id)
                ->with('success', 'تم استلام المنتجات وتحديث المخزون بنجاح');

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'حدث خطأ: ' . $e->getMessage());
        }
    }
}
