<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\MaintenanceRecord;
use App\Models\MaintenanceUsageItem;
use App\Models\Product;
use App\Models\Vehicle;
use App\Services\AccountingService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class MaintenanceRecordController extends Controller
{
    protected $accountingService;

    public function __construct(AccountingService $accountingService)
    {
        $this->accountingService = $accountingService;
    }

    public function create()
    {
        $vehicles = Vehicle::all();
        $products = Product::where('quantity', '>', 0)->get(); // Only show available parts
        return view('admin.maintenance.create', compact('vehicles', 'products'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'vehicle_id' => 'required|exists:vehicles,id',
            'title' => 'required|string',
            'start_date' => 'required|date',
            'items' => 'nullable|array',
            'items.*.product_id' => 'nullable|exists:products,id',
            'items.*.quantity' => 'required_with:items|integer|min:1',
        ]);

        try {
            DB::beginTransaction();

            $record = MaintenanceRecord::create([
                'vehicle_id' => $request->vehicle_id,
                'title' => $request->title,
                'description' => $request->description,
                'start_date' => $request->start_date,
                'status' => 'IN_PROGRESS',
                'technician' => $request->technician,
                'created_by' => auth()->id(),
                'cost' => 0,
            ]);

            $totalMaterialCost = 0;

            if ($request->items) {
                foreach ($request->items as $itemData) {
                    $qty = $itemData['quantity'];
                    $cost = 0;
                    $productId = $itemData['product_id'] ?? null;

                    if ($productId) {
                        $product = Product::find($productId);
                        
                        // Check if stock sufficient
                        if ($product->quantity < $qty) {
                            throw new \Exception("Insufficient stock for product: {$product->name}");
                        }

                        $product->decrement('quantity', $qty);
                        // Use cost_price if available, else price (as fallback/estimate)
                        $cost = $product->cost_price > 0 ? $product->cost_price : $product->price;
                    } else {
                        $cost = $itemData['unit_cost'] ?? 0;
                    }

                    $totalCost = $cost * $qty;
                    $totalMaterialCost += $totalCost;

                    MaintenanceUsageItem::create([
                        'maintenance_record_id' => $record->id,
                        'product_id' => $productId,
                        'description' => $itemData['description'] ?? ($product->name ?? 'Custom Item'),
                        'quantity' => $qty,
                        'unit_cost' => $cost,
                        'total_cost' => $totalCost,
                        'is_stock_deducted' => $productId ? true : false,
                    ]);
                }
            }

            $record->update(['cost' => $totalMaterialCost]);

            // Post COGS to Accounting
            if ($totalMaterialCost > 0) {
                $this->accountingService->postMaintenanceCost($record);
            }

            DB::commit();

            return redirect()->route('maintenance-admin')->with('success', 'Maintenance record created and parts deducted.');

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Error: ' . $e->getMessage())->withInput();
        }
    }
}
