<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

class OperationsController extends Controller
{
    // Calendar
    public function calendar(Request $request)
    {
        $events = \App\Models\CalendarEvent::when($request->month, function ($q) use ($request) {
            $q->whereMonth('start_time', $request->month);
        }, function ($q) {
            $q->whereMonth('start_time', now()->month);
        })->orderBy('start_time')->get();

        return view('admin.calendar.index', compact('events'));
    }

    public function calendarStore(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'start_time' => 'required|date',
            'end_time' => 'nullable|date|after:start_time',
            'type' => 'nullable|string',
            'description' => 'nullable|string',
            'all_day' => 'boolean',
        ]);
        \App\Models\CalendarEvent::create(array_merge($validated, ['created_by' => auth()->id()]));

        return back()->with('success', 'Event created.');
    }

    // Contracts
    public function contracts(Request $request)
    {
        $contracts = \App\Models\Contract::with(['customer', 'vehicle'])
            ->when($request->search, fn ($q, $s) => $q->where('contract_number', 'like', "%{$s}%"))
            ->when($request->status && $request->status !== 'all', fn ($q) => $q->where('status', request('status')))
            ->latest()->paginate(15);

        return view('admin.contracts.index', compact('contracts'));
    }

    public function contractStore(Request $request)
    {
        $validated = $request->validate([
            'customer_id' => 'required|exists:users,id',
            'vehicle_id' => 'nullable|exists:vehicles,id',
            'type' => 'required|string',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after:start_date',
            'value' => 'required|numeric|min:0',
            'terms' => 'nullable|string',
        ]);
        \App\Models\Contract::create(array_merge($validated, [
            'contract_number' => 'CTR-'.now()->format('Ymd').'-'.strtoupper(\Illuminate\Support\Str::random(6)),
            'status' => 'DRAFT',
            'created_by' => auth()->id(),
        ]));

        return back()->with('success', 'Contract created.');
    }

    // Insurance
    public function insurance(Request $request)
    {
        $policies = \App\Models\InsurancePolicy::with(['vehicle', 'customer', 'company'])
            ->when($request->search, fn ($q, $s) => $q->where('policy_number', 'like', "%{$s}%"))
            ->when($request->status && $request->status !== 'all', fn ($q) => $q->where('status', request('status')))
            ->latest()->paginate(15);
        $companies = \App\Models\InsuranceCompany::where('is_active', true)->pluck('name', 'id');

        return view('admin.insurance.index', compact('policies', 'companies'));
    }

    // Maintenance
    public function maintenance(Request $request)
    {
        $schedules = \App\Models\MaintenanceSchedule::with('vehicle')
            ->when($request->search, fn ($q, $s) => $q->where('title', 'like', "%{$s}%"))
            ->latest()->paginate(15);
        $records = \App\Models\MaintenanceRecord::with('vehicle')
            ->latest('start_date')->take(10)->get();

        return view('admin.maintenance.index', compact('schedules', 'records'));
    }
}
