<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Quotation;
use App\Models\QuotationItem;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class QuotationController extends Controller
{
    public function index()
    {
        $quotations = Quotation::with('customer')
            ->latest()
            ->paginate(15);

        return view('admin.quotations.index', compact('quotations'));
    }

    public function create()
    {
        $customers = \App\Models\User::where('role', 'CUSTOMER')->get();
        $vehicles = \App\Models\Vehicle::where('status', 'AVAILABLE')->get();

        return view('admin.quotations.create', compact('customers', 'vehicles'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'customer_id' => 'required|exists:users,id',
            'vehicle_id' => 'nullable|exists:vehicles,id',
            'valid_until' => 'required|date|after:today',
            'notes' => 'nullable|string|max:2000',
            'items' => 'required|array|min:1',
            'items.*.description' => 'required|string|max:255',
            'items.*.quantity' => 'required|integer|min:1',
            'items.*.unit_price' => 'required|numeric|min:0',
        ]);

        $quotation = Quotation::create([
            'quotation_number' => 'QT-'.now()->format('Ymd').'-'.strtoupper(Str::random(4)),
            'customer_id' => $validated['customer_id'],
            'vehicle_id' => $validated['vehicle_id'] ?? null,
            'valid_until' => $validated['valid_until'],
            'issue_date' => now(),
            'status' => 'DRAFT',
            'notes' => $validated['notes'] ?? null,
            'subtotal' => 0,
            'tax_amount' => 0,
            'total_amount' => 0,
            'created_by_id' => auth()->id(),
        ]);

        $subtotal = 0;
        foreach ($validated['items'] as $item) {
            $total = $item['quantity'] * $item['unit_price'];
            $subtotal += $total;

            QuotationItem::create([
                'quotation_id' => $quotation->id,
                'description' => $item['description'],
                'quantity' => $item['quantity'],
                'unit_price' => $item['unit_price'],
                'total' => $total,
            ]);
        }

        $taxAmount = $subtotal * 0.14; // 14% VAT
        $quotation->update([
            'subtotal' => $subtotal,
            'tax_amount' => $taxAmount,
            'total_amount' => $subtotal + $taxAmount,
        ]);

        return redirect()->route('admin.quotations.show', $quotation)
            ->with('success', 'تم إنشاء عرض السعر بنجاح');
    }

    public function show(Quotation $quotation)
    {
        $quotation->load(['customer', 'items', 'vehicle']);

        return view('admin.quotations.show', compact('quotation'));
    }

    public function convertToInvoice(Quotation $quotation)
    {
        if ($quotation->status === 'CONVERTED') {
            return back()->with('error', 'تم تحويل عرض السعر مسبقاً');
        }

        $invoice = \App\Models\Invoice::create([
            'invoice_number' => 'INV-'.now()->format('Ymd').'-'.strtoupper(Str::random(4)),
            'customer_id' => $quotation->customer_id,
            'vehicle_id' => $quotation->vehicle_id,
            'type' => 'SALE',
            'status' => 'DRAFT',
            'payment_status' => 'PENDING',
            'issue_date' => now(),
            'due_date' => now()->addDays(30),
            'subtotal' => $quotation->subtotal,
            'tax_amount' => $quotation->tax_amount,
            'total_amount' => $quotation->total_amount,
            'created_by' => auth()->id(),
        ]);

        $quotation->update(['status' => 'CONVERTED']);

        return redirect()->route('admin.finance.show', $invoice)
            ->with('success', 'تم تحويل عرض السعر إلى فاتورة بنجاح');
    }

    public function destroy(Quotation $quotation)
    {
        if ($quotation->status === 'CONVERTED') {
            return back()->with('error', 'لا يمكن حذف عرض سعر محوّل');
        }

        $quotation->items()->delete();
        $quotation->delete();

        return redirect()->route('admin.quotations.index')
            ->with('success', 'تم حذف عرض السعر');
    }
}
