<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

class RoleController extends Controller
{
    public function index()
    {
        $templates = \App\Models\RoleTemplate::withCount(['permissionsRel', 'users'])
            ->orderBy('role')
            ->orderBy('is_system', 'desc')
            ->get();

        return view('admin.roles.index', compact('templates'));
    }

    public function create()
    {
        $permissions = \App\Models\Permission::all()->groupBy('category');

        return view('admin.roles.create', compact('permissions'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255|unique:role_templates,name',
            'description' => 'nullable|string',
            'role' => 'required|string',
            'permissions' => 'nullable|array',
            'permissions.*' => 'exists:permissions,name', // Validating against permission names
        ]);

        $validated['is_active'] = true;
        // Also store permissions names in JSON column as per Next.js logic
        $validated['permissions'] = $validated['permissions'] ?? [];

        $template = \App\Models\RoleTemplate::create($validated);

        if (! empty($validated['permissions'])) {
            $permissionIds = \App\Models\Permission::whereIn('name', $validated['permissions'])->pluck('id');
            $template->permissionsRel()->sync($permissionIds);
        }

        return redirect()->route('admin.roles.index')->with('success', 'Role template created successfully.');
    }

    public function edit(\App\Models\RoleTemplate $role)
    {
        // $role is bound to route parameter {role}
        $permissions = \App\Models\Permission::all()->groupBy('category');
        $role->load('permissionsRel');

        return view('admin.roles.edit', compact('role', 'permissions'));
    }

    public function update(Request $request, \App\Models\RoleTemplate $role)
    {
        if ($role->is_system && $request->role !== $role->role) {
            return back()->withErrors(['role' => 'Cannot change role of system template.']);
        }

        $validated = $request->validate([
            'name' => 'required|string|max:255|unique:role_templates,name,'.$role->id,
            'description' => 'nullable|string',
            'role' => 'required|string',
            'permissions' => 'nullable|array',
            'permissions.*' => 'exists:permissions,name',
            'is_active' => 'boolean',
        ]);

        $validated['permissions'] = $validated['permissions'] ?? [];

        $role->update($validated);

        if (isset($validated['permissions'])) {
            $permissionIds = \App\Models\Permission::whereIn('name', $validated['permissions'])->pluck('id');
            $role->permissionsRel()->sync($permissionIds);
        }

        return redirect()->route('admin.roles.index')->with('success', 'Role template updated successfully.');
    }

    public function destroy(\App\Models\RoleTemplate $role)
    {
        if ($role->is_system) {
            return back()->with('error', 'Cannot delete system role template.');
        }

        if ($role->users()->count() > 0) {
            return back()->with('error', 'Cannot delete role template with assigned users.');
        }

        $role->delete();

        return redirect()->route('admin.roles.index')->with('success', 'Role template deleted successfully.');
    }
}
