<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

class SupportController extends Controller
{
    // Tickets
    public function index(Request $request)
    {
        $query = \App\Models\Ticket::with(['customer', 'assignedTo']);

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('ticket_number', 'like', "%{$search}%")
                    ->orWhere('subject', 'like', "%{$search}%")
                    ->orWhereHas('customer', function ($customerQuery) use ($search) {
                        $customerQuery->where('name', 'like', "%{$search}%");
                    });
            });
        }

        if ($request->filled('status') && $request->status !== 'all') {
            $query->where('status', $request->status);
        }

        if ($request->filled('priority') && $request->priority !== 'all') {
            $query->where('priority', $request->priority);
        }

        $tickets = $query->latest()->paginate(10);

        return view('admin.support.tickets.index', compact('tickets'));
    }

    public function show($id)
    {
        $ticket = \App\Models\Ticket::with(['customer', 'comments.author', 'assignedTo'])->findOrFail($id);

        return view('admin.support.tickets.show', compact('ticket'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'customer_id' => 'required|exists:users,id',
            'subject' => 'required|string',
            'description' => 'required|string',
            'category' => 'required|string',
            'priority' => 'required|string',
        ]);

        $ticket = \App\Models\Ticket::create(array_merge($validated, [
            'ticket_number' => 'TCK-'.now()->format('Ymd').'-'.strtoupper(\Illuminate\Support\Str::random(6)),
            'status' => 'OPEN',
            'branch_id' => auth()->user()->branch_id,
        ]));

        return back()->with('success', 'Ticket created successfully.');
    }

    public function storeComment(Request $request, $id)
    {
        $validated = $request->validate([
            'content' => 'required|string',
            'is_internal' => 'boolean',
        ]);

        \App\Models\TicketComment::create([
            'ticket_id' => $id,
            'author_id' => auth()->id(),
            'content' => $validated['content'],
            'is_internal' => $validated['is_internal'] ?? false,
        ]);

        return back()->with('success', 'Comment added successfully.');
    }

    public function updateStatus(Request $request, $id)
    {
        $ticket = \App\Models\Ticket::findOrFail($id);

        $validated = $request->validate([
            'status' => 'required|string',
        ]);

        $ticket->update(['status' => $validated['status']]);

        if ($validated['status'] === 'RESOLVED' || $validated['status'] === 'CLOSED') {
            $ticket->update([
                'resolved_at' => now(),
                'resolution_by' => auth()->id(),
            ]);
        }

        return back()->with('success', 'Ticket status updated.');
    }

    // Complaints
    public function complaintsIndex(Request $request)
    {
        $query = \App\Models\Complaint::with(['customer', 'assignedTo']);

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('complaint_number', 'like', "%{$search}%")
                    ->orWhere('subject', 'like', "%{$search}%")
                    ->orWhereHas('customer', function ($customerQuery) use ($search) {
                        $customerQuery->where('name', 'like', "%{$search}%");
                    });
            });
        }

        if ($request->filled('status') && $request->status !== 'all') {
            $query->where('status', $request->status);
        }

        if ($request->filled('priority') && $request->priority !== 'all') {
            $query->where('priority', $request->priority);
        }

        $complaints = $query->latest()->paginate(10);

        return view('admin.support.complaints.index', compact('complaints'));
    }

    public function complaintShow($id)
    {
        $complaint = \App\Models\Complaint::with(['customer', 'followUps.author', 'assignedTo'])->findOrFail($id);

        return view('admin.support.complaints.show', compact('complaint'));
    }

    public function complaintFollowUpStore(Request $request, $id)
    {
        $validated = $request->validate([
            'content' => 'required|string',
            'action' => 'required|string',
            'next_follow_up' => 'nullable|date',
        ]);

        \App\Models\ComplaintFollowUp::create([
            'complaint_id' => $id,
            'author_id' => auth()->id(),
            'content' => $validated['content'],
            'action' => $validated['action'],
            'next_follow_up' => $validated['next_follow_up'],
        ]);

        return back()->with('success', 'Follow-up added successfully.');
    }
}
