<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\URL;

class SystemController extends Controller
{
    // Activity Logs
    public function logs(Request $request)
    {
        $logs = \App\Models\AuditLog::with('user')
            ->when($request->search, fn ($q, $s) => $q->where(fn ($inner) => $inner->where('model_type', 'like', "%{$s}%")->orWhere('action', 'like', "%{$s}%")))
            ->when($request->type && $request->type !== 'all', fn ($q) => $q->where('action', request('type')))
            ->latest()->paginate(20);

        return view('admin.logs.index', compact('logs'));
    }

    // Security Logs
    public function securityLogs(Request $request)
    {
        $logs = \App\Models\SecurityLog::with('user')
            ->when($request->severity && $request->severity !== 'all', fn ($q) => $q->where('severity', request('severity')))
            ->latest()->paginate(20);

        return view('admin.logs.security', compact('logs'));
    }

    // Notifications
    public function notifications(Request $request)
    {
        $notifications = auth()->user()->notifications()->paginate(20);

        return view('admin.notifications.index', compact('notifications'));
    }

    public function markNotificationRead($id)
    {
        auth()->user()->notifications()->where('id', $id)->update(['read_at' => now()]);

        return back()->with('success', 'Marked as read.');
    }

    // Media Library
    public function media(Request $request)
    {
        $media = \App\Models\Media::when($request->search, fn ($q, $s) => $q->where(fn ($inner) => $inner->where('original_name', 'like', "%{$s}%")->orWhere('title', 'like', "%{$s}%")))
            ->when($request->type && $request->type !== 'all', fn ($q) => $q->where('mime_type', 'like', request('type').'%'))
            ->latest()->paginate(24);

        $media->getCollection()->transform(function ($item) {
            if (! $item->is_public) {
                $item->url = URL::temporarySignedRoute('admin.media.download', now()->addMinutes(10), ['media' => $item->id]);
            }

            return $item;
        });

        return view('admin.media.index', compact('media'));
    }

    public function mediaStore(Request $request)
    {
        $request->validate([
            'file' => 'required|file|max:10240|mimes:jpg,jpeg,png,gif,webp,pdf,doc,docx,xls,xlsx,mp4,mov,avi|mimetypes:image/jpeg,image/png,image/gif,image/webp,application/pdf,application/msword,application/vnd.openxmlformats-officedocument.wordprocessingml.document,application/vnd.ms-excel,application/vnd.openxmlformats-officedocument.spreadsheetml.sheet,video/mp4,video/quicktime,video/x-msvideo',
        ]);

        $file = $request->file('file');
        $sensitiveExtensions = ['pdf', 'doc', 'docx', 'xls', 'xlsx'];
        $isSensitive = in_array(strtolower($file->getClientOriginalExtension()), $sensitiveExtensions, true);

        $disk = $isSensitive ? 'sensitive' : 'public';
        $directory = $isSensitive ? 'media-private' : 'media';
        $path = $file->store($directory, $disk);

        \App\Models\Media::create([
            'filename' => basename($path),
            'original_name' => $file->getClientOriginalName(),
            'path' => $path,
            'url' => $isSensitive ? '#' : '/storage/'.$path,
            'mime_type' => $file->getMimeType(),
            'size' => $file->getSize(),
            'is_public' => ! $isSensitive,
            'created_by' => auth()->id(),
        ]);

        return back()->with('success', 'File uploaded.');
    }

    public function mediaDownload(Request $request, \App\Models\Media $media)
    {
        abort_unless($request->hasValidSignature(), 403);
        abort_if($media->is_public, 404);

        if (! Storage::disk('sensitive')->exists($media->path)) {
            abort(404, 'File not found.');
        }

        return Storage::disk('sensitive')->download($media->path, $media->original_name);
    }

    // Tasks
    public function tasks(Request $request)
    {
        $tasks = \App\Models\Task::with(['assignee', 'assigner'])
            ->when($request->status && $request->status !== 'all', fn ($q) => $q->where('status', request('status')))
            ->latest()->paginate(15);

        return view('admin.tasks.index', compact('tasks'));
    }

    public function taskStore(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'assigned_to' => 'nullable|exists:users,id',
            'priority' => 'required|in:low,medium,high,urgent',
            'due_date' => 'nullable|date',
        ]);
        \App\Models\Task::create(array_merge($validated, ['assigned_by' => auth()->id(), 'status' => 'PENDING']));

        return back()->with('success', 'Task created.');
    }
}
