<?php

namespace App\Http\Requests\Admin\Cms;

use Illuminate\Foundation\Http\FormRequest;

class UpdateSiteSettingsRequest extends FormRequest
{
    private const ALLOWED_SOCIAL_KEYS = [
        'facebook',
        'instagram',
        'twitter',
        'youtube',
        'linkedin',
        'tiktok',
        'whatsapp',
    ];

    public function authorize(): bool
    {
        return $this->user()?->can('content.manage') ?? false;
    }

    protected function prepareForValidation(): void
    {
        $incoming = (array) $this->input('social_links', []);
        $filtered = [];

        foreach (self::ALLOWED_SOCIAL_KEYS as $key) {
            $value = $incoming[$key] ?? null;

            if (! is_string($value)) {
                continue;
            }

            $value = trim($value);
            if ($value === '') {
                continue;
            }

            if ($key === 'whatsapp' && preg_match('/^\+?\d+$/', $value)) {
                $value = 'https://wa.me/'.preg_replace('/\D+/', '', $value);
            } elseif (! preg_match('/^https:\/\//i', $value)) {
                $value = 'https://'.ltrim($value, '/');
            }

            $filtered[$key] = $value;
        }

        $this->merge(['social_links' => $filtered]);
    }

    public function rules(): array
    {
        return [
            'site_title' => ['required', 'string', 'max:255'],
            'site_description' => ['nullable', 'string'],
            'logo_url' => ['nullable', 'string', 'max:255'],
            'favicon_url' => ['nullable', 'string', 'max:255'],
            'contact_email' => ['nullable', 'email', 'max:255'],
            'contact_phone' => ['nullable', 'string', 'max:50'],
            'contact_address' => ['nullable', 'string', 'max:500'],
            'primary_color' => ['nullable', 'string', 'max:20'],
            'social_links' => ['nullable', 'array'],
            'social_links.*' => ['nullable', 'url', 'regex:/^https:\/\//i', 'max:255'],
        ];
    }
}
