<?php

namespace App\Providers;

use App\Enums\UserRole;
use App\Models\SiteSetting;
use Illuminate\Cache\RateLimiting\Limit;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\RateLimiter;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\View;
use Illuminate\Support\ServiceProvider;
use stdClass;

class AppServiceProvider extends ServiceProvider
{
    public function register(): void
    {
        //
    }

    public function boot(): void
    {
        Route::model('role', \App\Models\RoleTemplate::class);

        Gate::policy(\App\Models\User::class, \App\Policies\UserPolicy::class);
        Gate::policy(\App\Models\Invoice::class, \App\Policies\InvoicePolicy::class);
        Gate::policy(\App\Models\JournalEntry::class, \App\Policies\JournalEntryPolicy::class);
        $this->registerModuleGates();

        RateLimiter::for('contact', fn (Request $request) => Limit::perMinute(5)->by($request->ip()));
        RateLimiter::for('public-action', fn (Request $request) => Limit::perMinute(10)->by($request->user()?->id ?: $request->ip()));
        RateLimiter::for('api', fn (Request $request) => Limit::perMinute(60)->by($request->user()?->id ?: $request->ip()));
        RateLimiter::for('search', fn (Request $request) => Limit::perMinute(30)->by($request->user()?->id ?: $request->ip()));

        View::composer('*', function ($view): void {
            $view->with('siteSettings', $this->resolveSiteSettings());
        });

        \App\Models\LeadActivity::observe(\App\Observers\LeadObserver::class);

        // Audit Logs
        $auditObserver = \App\Observers\AuditObserver::class;
        \App\Models\User::observe($auditObserver);
        // Employee might not be a model or might be App\Models\Employee
        if (class_exists(\App\Models\Employee::class)) {
            \App\Models\Employee::observe($auditObserver);
        }
        if (class_exists(\App\Models\PurchaseOrder::class)) {
            \App\Models\PurchaseOrder::observe($auditObserver);
        }
        if (class_exists(\App\Models\JournalEntry::class)) {
            \App\Models\JournalEntry::observe($auditObserver);
        }

        View::composer('components.public-footer', function ($view): void {
            $view->with($this->resolveFooterData());
        });
    }

    private function registerModuleGates(): void
    {
        $this->defineRoleGate('users.manage', [UserRole::SuperAdmin, UserRole::Admin]);
        $this->defineRoleGate('finance.manage', [UserRole::SuperAdmin, UserRole::Admin, UserRole::FinancialAccountant]);
        $this->defineRoleGate('content.manage', [UserRole::SuperAdmin, UserRole::Admin, UserRole::ContentEditor]);
        $this->defineRoleGate('support.manage', [UserRole::SuperAdmin, UserRole::Admin, UserRole::BranchManager, UserRole::SalesManager]);
        $this->defineRoleGate('hr.manage', [UserRole::SuperAdmin, UserRole::Admin, UserRole::HRManager, UserRole::BranchManager]);
        $this->defineRoleGate('crm.manage', [UserRole::SuperAdmin, UserRole::Admin, UserRole::SalesManager, UserRole::BranchManager]);
        $this->defineRoleGate('operations.manage', [UserRole::SuperAdmin, UserRole::Admin, UserRole::BranchManager]);
        $this->defineRoleGate('system.manage', [UserRole::SuperAdmin, UserRole::Admin]);
    }

    private function defineRoleGate(string $ability, array $roles): void
    {
        $allowedRoleValues = array_map(fn (UserRole $role) => $role->value, $roles);

        Gate::define($ability, fn (\App\Models\User $user): bool => in_array($user->role, $allowedRoleValues, true));
    }

    private function resolveSiteSettings(): object
    {
        return Cache::remember('site_settings', now()->addHour(), function (): object {
            if (! Schema::hasTable('site_settings')) {
                return $this->defaultSiteSettings();
            }

            try {
                return SiteSetting::first() ?? $this->defaultSiteSettings();
            } catch (\Throwable) {
                return $this->defaultSiteSettings();
            }
        });
    }

    private function resolveFooterData(): array
    {
        return Cache::remember('footer_data', now()->addMinutes(30), function (): array {
            try {
                $companyInfo = Schema::hasTable('company_infos')
                    ? \App\Models\CompanyInfo::select('description')->first()
                    : null;

                $vehicles = Schema::hasTable('vehicles')
                    ? \App\Models\Vehicle::where('status', 'AVAILABLE')->select('id', 'make', 'model')->latest()->take(5)->get()
                    : collect();

                $services = Schema::hasTable('service_items')
                    ? \App\Models\ServiceItem::select('id', 'name')->where('is_active', true)->orderBy('order')->take(5)->get()
                    : collect();

                return [
                    'companyInfo' => $companyInfo,
                    'vehicles' => $vehicles,
                    'services' => $services,
                ];
            } catch (\Throwable) {
                return [
                    'companyInfo' => null,
                    'vehicles' => collect(),
                    'services' => collect(),
                ];
            }
        });
    }

    private function defaultSiteSettings(): stdClass
    {
        return (object) [
            'site_title' => 'الحمد للسيارات',
            'site_description' => 'الموزع المعتمد لسيارات تاتا موتورز',
            'logo_url' => '/uploads/logo/elhamd-logo.png',
            'contact_email' => 'info@elhamdimport.online',
            'contact_phone' => '+201555767729',
            'contact_address' => 'بورسعيد - الحي الإماراتي',
            'social_links' => [],
        ];
    }
}
