<?php

namespace App\Traits;

use App\Models\ActivityLog;

trait LogsActivity
{
    public static function bootLogsActivity(): void
    {
        static::created(function ($model) {
            static::logActivity($model, 'created');
        });

        static::updated(function ($model) {
            $changedKeys = array_keys($model->getDirty());
            if (! empty($changedKeys)) {
                static::logActivity($model, 'updated', $model->getOriginal(), $model->getAttributes(), $changedKeys);
            }
        });

        static::deleted(function ($model) {
            static::logActivity($model, 'deleted');
        });

        if (method_exists(static::class, 'restored')) {
            static::restored(function ($model) {
                static::logActivity($model, 'restored');
            });
        }
    }

    protected static function logActivity($model, string $action, ?array $oldValues = null, ?array $newValues = null, ?array $changedKeys = null): void
    {
        try {
            $user = auth()->user();
            $request = request();
            $changedLookup = $changedKeys ? array_flip($changedKeys) : null;

            ActivityLog::create([
                'user_id' => $user?->id,
                'action' => $action,
                'model_type' => get_class($model),
                'model_id' => $model->getKey(),
                'description' => $action.' '.class_basename($model).' #'.$model->getKey(),
                'old_values' => $oldValues ? ($changedLookup ? array_intersect_key($oldValues, $changedLookup) : $oldValues) : null,
                'new_values' => $newValues ? ($changedLookup ? array_intersect_key($newValues, $changedLookup) : $newValues) : null,
                'ip_address' => $request?->ip(),
                'user_agent' => $request?->userAgent(),
            ]);
        } catch (\Throwable $e) {
            // Silently fail - logging should never break the app
            logger()->warning('Activity log failed: '.$e->getMessage());
        }
    }
}
