<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        Schema::create('branches', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->string('name');
            $table->string('code')->unique();
            $table->string('address')->nullable();
            $table->string('phone')->nullable();
            $table->string('email')->nullable();
            $table->uuid('manager_id')->nullable(); // Foreign key added later
            $table->boolean('is_active')->default(true);
            $table->dateTime('opening_date')->nullable();
            $table->string('currency')->default('EGP');
            $table->string('timezone')->default('Africa/Cairo');
            $table->float('map_lat')->nullable();
            $table->float('map_lng')->nullable();
            $table->string('google_map_link')->nullable();
            $table->json('settings')->nullable();
            $table->timestamps();
        });

        Schema::create('permissions', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->string('name')->unique();
            $table->string('description')->nullable();
            $table->string('category');
            $table->boolean('is_active')->default(true);
            $table->timestamps();
        });

        Schema::create('role_templates', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->string('name')->unique();
            $table->string('description')->nullable();
            $table->string('role'); // Enum UserRole
            $table->json('permissions');
            $table->boolean('is_active')->default(true);
            $table->boolean('is_system')->default(false);
            $table->timestamps();
        });

        Schema::create('role_template_permissions', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->uuid('template_id');
            $table->uuid('permission_id');
            $table->timestamps();

            $table->foreign('template_id')->references('id')->on('role_templates')->onDelete('cascade');
            $table->foreign('permission_id')->references('id')->on('permissions')->onDelete('cascade');
            $table->unique(['template_id', 'permission_id']);
        });

        Schema::create('user_permissions', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->foreignUuid('user_id')->constrained()->cascadeOnDelete();
            $table->uuid('permission_id');
            $table->uuid('granted_by')->nullable();
            $table->dateTime('granted_at')->useCurrent();
            $table->dateTime('expires_at')->nullable();

            $table->foreign('permission_id')->references('id')->on('permissions')->onDelete('cascade');
            $table->unique(['user_id', 'permission_id']);
        });

        Schema::create('branch_permissions', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->foreignUuid('user_id')->constrained()->cascadeOnDelete();
            $table->uuid('branch_id');
            $table->json('permissions');
            $table->uuid('granted_by')->nullable();
            $table->dateTime('granted_at')->useCurrent();
            $table->dateTime('expires_at')->nullable();
            $table->boolean('is_active')->default(true);
            $table->timestamps();

            $table->foreign('branch_id')->references('id')->on('branches')->onDelete('cascade');
            $table->unique(['user_id', 'branch_id']);
        });

        // Add Foreign Keys to Users table
        Schema::table('users', function (Blueprint $table) {
            if (Schema::hasColumn('users', 'branch_id')) {
                $table->foreign('branch_id')->references('id')->on('branches')->nullOnDelete();
            }
            if (Schema::hasColumn('users', 'role_template_id')) {
                $table->foreign('role_template_id')->references('id')->on('role_templates')->nullOnDelete();
            }
        });
    }

    public function down(): void
    {
        Schema::table('users', function (Blueprint $table) {
            $table->dropForeign(['branch_id']);
            $table->dropForeign(['role_template_id']);
        });
        Schema::dropIfExists('branch_permissions');
        Schema::dropIfExists('user_permissions');
        Schema::dropIfExists('role_template_permissions');
        Schema::dropIfExists('role_templates');
        Schema::dropIfExists('permissions');
        Schema::dropIfExists('branches');
    }
};
