<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        Schema::create('invoices', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->string('invoice_number')->unique();
            $table->uuid('customer_id');
            $table->uuid('branch_id')->nullable();
            $table->uuid('vehicle_id')->nullable();
            $table->string('type')->default('SERVICE'); // Enum InvoiceType
            $table->string('status')->default('DRAFT'); // Enum InvoiceStatus
            $table->string('payment_status')->default('PENDING'); // Enum InvoicePaymentStatus
            $table->dateTime('issue_date');
            $table->dateTime('due_date');
            $table->float('subtotal')->default(0);
            $table->float('tax_amount')->default(0);
            $table->float('total_amount')->default(0);
            $table->float('paid_amount')->default(0);
            $table->string('currency')->default('EGP');
            $table->text('notes')->nullable();
            $table->text('terms')->nullable();
            $table->uuid('created_by')->nullable(); // Employee or User ID
            $table->boolean('is_deleted')->default(false);
            $table->timestamps();

            $table->foreign('customer_id')->references('id')->on('users');
            $table->foreign('branch_id')->references('id')->on('branches')->nullOnDelete();
            $table->foreign('vehicle_id')->references('id')->on('vehicles')->nullOnDelete();
        });

        Schema::create('invoice_items', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->uuid('invoice_id');
            $table->uuid('vehicle_id')->nullable();
            $table->uuid('inventory_item_id')->nullable(); // Foreign key to inventory (later)
            $table->text('description');
            $table->integer('quantity')->default(1);
            $table->float('unit_price')->default(0);
            $table->float('total_price')->default(0);
            $table->float('tax_rate')->default(0);
            $table->float('tax_amount')->default(0);
            $table->timestamps();

            $table->foreign('invoice_id')->references('id')->on('invoices')->onDelete('cascade');
            $table->foreign('vehicle_id')->references('id')->on('vehicles')->nullOnDelete();
        });

        Schema::create('payments', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->uuid('booking_id')->nullable(); // Generic booking link
            $table->string('booking_type')->default('SERVICE'); // Enum BookingType
            $table->float('amount');
            $table->string('currency')->default('EGP');
            $table->string('status')->default('PENDING'); // Enum PaymentStatus
            $table->string('payment_method'); // Enum PaymentMethod
            $table->string('transaction_id')->nullable();
            $table->string('receipt_url')->nullable();
            $table->text('description')->nullable();
            $table->text('notes')->nullable();
            $table->uuid('branch_id')->nullable();
            $table->uuid('customer_id')->nullable();
            $table->uuid('invoice_id')->nullable();
            $table->uuid('service_booking_id')->nullable();
            $table->uuid('test_drive_booking_id')->nullable();
            $table->timestamps();

            $table->foreign('branch_id')->references('id')->on('branches')->nullOnDelete();
            $table->foreign('customer_id')->references('id')->on('users');
            $table->foreign('invoice_id')->references('id')->on('invoices')->nullOnDelete();
            $table->foreign('service_booking_id')->references('id')->on('service_bookings')->nullOnDelete();
            $table->foreign('test_drive_booking_id')->references('id')->on('test_drive_bookings')->nullOnDelete();
        });

        Schema::create('transactions', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->string('reference_id')->unique();
            $table->uuid('branch_id')->nullable();
            $table->string('type')->default('INCOME');
            $table->string('category');
            $table->float('amount');
            $table->string('currency')->default('EGP');
            $table->text('description')->nullable();
            $table->dateTime('date');
            $table->string('payment_method');
            $table->string('reference')->nullable();
            $table->uuid('customer_id')->nullable();
            $table->uuid('invoice_id')->nullable();
            $table->timestamps();

            $table->foreign('branch_id')->references('id')->on('branches')->nullOnDelete();
            $table->foreign('customer_id')->references('id')->on('users')->nullOnDelete();
            $table->foreign('invoice_id')->references('id')->on('invoices')->nullOnDelete();
        });

        Schema::create('quotations', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->string('quotation_number')->unique();
            $table->uuid('customer_id');
            $table->uuid('branch_id')->nullable();
            $table->string('status')->default('DRAFT');
            $table->dateTime('issue_date');
            $table->dateTime('valid_until');
            $table->float('subtotal')->default(0);
            $table->float('tax_amount')->default(0);
            $table->float('total_amount')->default(0);
            $table->string('currency')->default('EGP');
            $table->text('notes')->nullable();
            $table->text('terms')->nullable();
            $table->uuid('created_by_id');
            $table->uuid('converted_to_invoice_id')->nullable()->unique();
            $table->timestamps();

            $table->foreign('customer_id')->references('id')->on('users');
            $table->foreign('branch_id')->references('id')->on('branches')->nullOnDelete();
            $table->foreign('created_by_id')->references('id')->on('users');
            $table->foreign('converted_to_invoice_id')->references('id')->on('invoices');
        });

        Schema::create('quotation_items', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->uuid('quotation_id');
            $table->text('description');
            $table->integer('quantity')->default(1);
            $table->float('unit_price')->default(0);
            $table->float('total_price')->default(0);
            $table->float('tax_rate')->default(0);
            $table->float('tax_amount')->default(0);
            $table->timestamps();

            $table->foreign('quotation_id')->references('id')->on('quotations')->onDelete('cascade');
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('quotation_items');
        Schema::dropIfExists('quotations');
        Schema::dropIfExists('transactions');
        Schema::dropIfExists('payments');
        Schema::dropIfExists('invoice_items');
        Schema::dropIfExists('invoices');
    }
};
