<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        Schema::create('chart_of_accounts', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->string('code')->unique();
            $table->string('name');
            $table->string('type'); // ASSET, LIABILITY, EQUITY, REVENUE, EXPENSE
            $table->uuid('parent_id')->nullable();
            $table->boolean('is_active')->default(true);
            $table->string('normal_balance'); // DEBIT, CREDIT
            $table->timestamps();
            $table->index('code');
        });

        Schema::create('journal_entries', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->string('entry_number')->unique();
            $table->dateTime('date');
            $table->text('description');
            $table->string('reference')->nullable();
            $table->float('total_debit');
            $table->float('total_credit');
            $table->string('status')->default('DRAFT'); // DRAFT, POSTED, VOID
            $table->uuid('created_by');
            $table->uuid('approved_by')->nullable();
            $table->dateTime('approved_at')->nullable();
            $table->uuid('branch_id')->nullable();
            $table->timestamps();
            $table->index('date');
            $table->foreign('created_by')->references('id')->on('users');
            $table->foreign('branch_id')->references('id')->on('branches')->nullOnDelete();
        });

        Schema::create('journal_entry_items', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->uuid('entry_id');
            $table->uuid('account_id');
            $table->text('description')->nullable();
            $table->float('debit')->default(0);
            $table->float('credit')->default(0);
            $table->timestamps();
            $table->foreign('entry_id')->references('id')->on('journal_entries')->onDelete('cascade');
            $table->foreign('account_id')->references('id')->on('chart_of_accounts');
            $table->index('entry_id');
            $table->index('account_id');
        });

        Schema::create('tax_rates', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->string('name');
            $table->string('type')->default('STANDARD'); // STANDARD, REDUCED, ZERO, EXEMPT
            $table->float('rate')->default(0);
            $table->text('description')->nullable();
            $table->boolean('is_active')->default(true);
            $table->dateTime('effective_from');
            $table->dateTime('effective_to')->nullable();
            $table->timestamps();
        });

        Schema::create('tax_records', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->string('type'); // VAT, INCOME, CORPORATE, WITHHOLDING, SALES, CUSTOMS
            $table->string('period');
            $table->float('amount');
            $table->dateTime('due_date');
            $table->dateTime('paid_date')->nullable();
            $table->string('status')->default('PENDING');
            $table->string('reference')->nullable();
            $table->json('documents')->nullable();
            $table->text('notes')->nullable();
            $table->uuid('branch_id')->nullable();
            $table->uuid('created_by');
            $table->uuid('approved_by')->nullable();
            $table->timestamps();
            $table->foreign('branch_id')->references('id')->on('branches')->nullOnDelete();
            $table->foreign('created_by')->references('id')->on('users');
            $table->index(['type', 'period']);
            $table->index('due_date');
            $table->index('status');
        });

        Schema::create('invoice_taxes', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->uuid('invoice_id');
            $table->uuid('tax_rate_id')->nullable();
            $table->string('tax_type');
            $table->float('rate')->default(0);
            $table->float('tax_amount')->default(0);
            $table->text('description')->nullable();
            $table->timestamps();
            $table->foreign('invoice_id')->references('id')->on('invoices')->onDelete('cascade');
            $table->foreign('tax_rate_id')->references('id')->on('tax_rates')->nullOnDelete();
        });

        Schema::create('invoice_installments', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->uuid('invoice_id');
            $table->integer('sequence');
            $table->float('amount');
            $table->dateTime('due_date');
            $table->string('status')->default('PENDING');
            $table->float('paid_amount')->default(0);
            $table->dateTime('paid_at')->nullable();
            $table->text('notes')->nullable();
            $table->uuid('payment_id')->nullable();
            $table->json('metadata')->nullable();
            $table->timestamps();
            $table->foreign('invoice_id')->references('id')->on('invoices')->onDelete('cascade');
            $table->foreign('payment_id')->references('id')->on('payments')->nullOnDelete();
            $table->unique(['invoice_id', 'sequence']);
        });

        Schema::create('invoice_payments', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->uuid('invoice_id');
            $table->uuid('payment_id');
            $table->float('amount');
            $table->dateTime('payment_date');
            $table->string('payment_method');
            $table->string('transaction_id')->nullable();
            $table->text('notes')->nullable();
            $table->timestamps();
            $table->foreign('invoice_id')->references('id')->on('invoices')->onDelete('cascade');
            $table->foreign('payment_id')->references('id')->on('payments');
            $table->unique(['invoice_id', 'payment_id']);
        });

        Schema::create('financial_reports', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->string('title');
            $table->string('type'); // INCOME_STATEMENT, BALANCE_SHEET, CASH_FLOW, etc.
            $table->text('description')->nullable();
            $table->json('parameters')->nullable();
            $table->dateTime('generated_at')->useCurrent();
            $table->string('generated_by');
            $table->json('data')->nullable();
            $table->string('file_url')->nullable();
            $table->string('status')->default('GENERATING');
            $table->timestamps();
        });

        Schema::create('payment_gateway_configs', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->string('name');
            $table->string('gateway'); // STRIPE, PAYPAL, FAWRY, etc.
            $table->string('api_key')->nullable();
            $table->string('secret_key')->nullable();
            $table->string('merchant_id')->nullable();
            $table->boolean('is_active')->default(true);
            $table->json('config')->nullable();
            $table->boolean('test_mode')->default(true);
            $table->timestamps();
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('payment_gateway_configs');
        Schema::dropIfExists('financial_reports');
        Schema::dropIfExists('invoice_payments');
        Schema::dropIfExists('invoice_installments');
        Schema::dropIfExists('invoice_taxes');
        Schema::dropIfExists('tax_records');
        Schema::dropIfExists('tax_rates');
        Schema::dropIfExists('journal_entry_items');
        Schema::dropIfExists('journal_entries');
        Schema::dropIfExists('chart_of_accounts');
    }
};
