<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        Schema::create('departments', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->string('name')->unique();
            $table->text('description')->nullable();
            $table->boolean('is_active')->default(true);
            $table->timestamps();
        });

        Schema::create('positions', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->string('title');
            $table->uuid('department_id');
            $table->string('level')->default('JUNIOR');
            $table->text('description')->nullable();
            $table->boolean('is_active')->default(true);
            $table->timestamps();
            $table->foreign('department_id')->references('id')->on('departments')->onDelete('cascade');
            $table->unique(['title', 'department_id']);
        });

        Schema::create('employees', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->string('employee_number')->unique();
            $table->uuid('user_id')->unique();
            $table->uuid('department_id')->nullable();
            $table->uuid('position_id')->nullable();
            $table->dateTime('hire_date');
            $table->float('salary');
            $table->string('bank_account')->nullable();
            $table->string('tax_number')->nullable();
            $table->string('insurance_number')->nullable();
            $table->string('status')->default('ACTIVE');
            $table->uuid('manager_id')->nullable();
            $table->uuid('branch_id')->nullable();
            $table->json('emergency_contact')->nullable();
            $table->text('notes')->nullable();
            $table->uuid('payroll_expense_account_id')->nullable();
            $table->uuid('payroll_liability_account_id')->nullable();
            $table->timestamps();
            $table->foreign('user_id')->references('id')->on('users')->onDelete('cascade');
            $table->foreign('department_id')->references('id')->on('departments')->nullOnDelete();
            $table->foreign('position_id')->references('id')->on('positions')->nullOnDelete();
            $table->foreign('manager_id')->references('id')->on('employees')->nullOnDelete();
            $table->foreign('branch_id')->references('id')->on('branches')->nullOnDelete();
            $table->index('employee_number');
            $table->index('department_id');
            $table->index('status');
        });

        Schema::create('attendance_records', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->uuid('employee_id');
            $table->date('date');
            $table->dateTime('check_in')->nullable();
            $table->dateTime('check_out')->nullable();
            $table->string('status')->default('PRESENT');
            $table->text('notes')->nullable();
            $table->timestamps();
            $table->foreign('employee_id')->references('id')->on('employees');
            $table->unique(['employee_id', 'date']);
            $table->index('date');
            $table->index('status');
        });

        Schema::create('performance_metrics', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->uuid('employee_id');
            $table->string('period'); // DAILY, WEEKLY, MONTHLY, QUARTERLY, YEARLY
            $table->string('period_label')->default('CURRENT');
            $table->integer('bookings_handled')->default(0);
            $table->float('average_handling_time')->default(0);
            $table->float('customer_rating')->default(0);
            $table->float('conversion_rate')->default(0);
            $table->float('revenue_generated')->default(0);
            $table->integer('tasks_completed')->default(0);
            $table->float('customer_satisfaction')->default(0);
            $table->float('response_time')->default(0);
            $table->float('follow_up_rate')->default(0);
            $table->float('upsell_success')->default(0);
            $table->float('overall_score')->default(0);
            $table->json('metrics_metadata')->nullable();
            $table->text('notes')->nullable();
            $table->timestamps();
            $table->foreign('employee_id')->references('id')->on('employees');
            $table->unique(['employee_id', 'period', 'period_label']);
        });

        Schema::create('performance_reviews', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->uuid('employee_id');
            $table->uuid('reviewer_id');
            $table->string('period');
            $table->string('rating'); // EXCELLENT, GOOD, SATISFACTORY, NEEDS_IMPROVEMENT, UNSATISFACTORY
            $table->json('goals')->nullable();
            $table->json('achievements')->nullable();
            $table->json('areas_for_improvement')->nullable();
            $table->text('comments')->nullable();
            $table->string('status')->default('DRAFT');
            $table->timestamps();
            $table->foreign('employee_id')->references('id')->on('employees')->onDelete('cascade');
            $table->foreign('reviewer_id')->references('id')->on('users');
            $table->index(['employee_id', 'period']);
        });

        Schema::create('training_records', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->uuid('employee_id');
            $table->string('training_name');
            $table->text('description')->nullable();
            $table->dateTime('start_date');
            $table->dateTime('end_date');
            $table->float('cost')->nullable();
            $table->string('provider')->nullable();
            $table->string('certificate')->nullable();
            $table->string('status')->default('PLANNED');
            $table->text('feedback')->nullable();
            $table->timestamps();
            $table->foreign('employee_id')->references('id')->on('employees')->onDelete('cascade');
            $table->index('employee_id');
        });

        Schema::create('salary_advances', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->uuid('employee_id');
            $table->uuid('requested_by');
            $table->float('amount');
            $table->string('status')->default('PENDING');
            $table->text('reason')->nullable();
            $table->dateTime('requested_at')->useCurrent();
            $table->uuid('approved_by')->nullable();
            $table->dateTime('approved_at')->nullable();
            $table->dateTime('disbursed_at')->nullable();
            $table->dateTime('repayment_start')->nullable();
            $table->integer('repayment_months')->nullable();
            $table->json('repayment_schedule')->nullable();
            $table->float('repaid_amount')->default(0);
            $table->dateTime('next_due_date')->nullable();
            $table->text('notes')->nullable();
            $table->json('metadata')->nullable();
            $table->timestamps();
            $table->foreign('employee_id')->references('id')->on('employees')->onDelete('cascade');
            $table->foreign('requested_by')->references('id')->on('users');
            $table->foreign('approved_by')->references('id')->on('users')->nullOnDelete();
            $table->index(['employee_id', 'status']);
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('salary_advances');
        Schema::dropIfExists('training_records');
        Schema::dropIfExists('performance_reviews');
        Schema::dropIfExists('performance_metrics');
        Schema::dropIfExists('attendance_records');
        Schema::dropIfExists('employees');
        Schema::dropIfExists('positions');
        Schema::dropIfExists('departments');
    }
};
