<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        Schema::create('customer_interactions', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->uuid('customer_id');
            $table->string('type'); // CALL, EMAIL, MEETING, SMS, SOCIAL_MEDIA, WEBSITE_VISIT, TEST_DRIVE, SERVICE_VISIT, PURCHASE, FOLLOW_UP
            $table->string('title');
            $table->text('description')->nullable();
            $table->uuid('performed_by')->nullable();
            $table->dateTime('interaction_date')->useCurrent();
            $table->string('outcome')->nullable();
            $table->text('notes')->nullable();
            $table->json('metadata')->nullable();
            $table->timestamps();
            $table->foreign('customer_id')->references('id')->on('customer_profiles')->onDelete('cascade');
            $table->foreign('performed_by')->references('id')->on('users')->nullOnDelete();
            $table->index(['customer_id', 'interaction_date']);
        });

        Schema::create('customer_journeys', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->uuid('customer_id');
            $table->string('event');
            $table->text('description')->nullable();
            $table->string('channel')->default('WEB');
            $table->json('metadata')->nullable();
            $table->timestamps();
            $table->foreign('customer_id')->references('id')->on('customer_profiles')->onDelete('cascade');
            $table->index('customer_id');
        });

        Schema::create('customer_lifecycles', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->uuid('customer_id');
            $table->string('stage')->default('AWARENESS');
            $table->string('sub_stage')->nullable();
            $table->dateTime('entry_date')->useCurrent();
            $table->dateTime('exit_date')->nullable();
            $table->integer('duration')->nullable();
            $table->integer('score')->default(0);
            $table->string('next_stage')->nullable();
            $table->json('actions')->nullable();
            $table->json('metadata')->nullable();
            $table->timestamps();
            $table->foreign('customer_id')->references('id')->on('customer_profiles')->onDelete('cascade');
            $table->unique(['customer_id', 'stage']);
        });

        Schema::create('customer_feedbacks', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->uuid('customer_id');
            $table->string('type')->default('GENERAL');
            $table->integer('rating')->default(0);
            $table->text('comment')->nullable();
            $table->string('status')->default('PENDING');
            $table->text('response')->nullable();
            $table->uuid('responded_by')->nullable();
            $table->dateTime('responded_at')->nullable();
            $table->timestamps();
            $table->foreign('customer_id')->references('id')->on('customer_profiles')->onDelete('cascade');
            $table->index('customer_id');
        });

        Schema::create('crm_interactions', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->uuid('lead_id')->nullable();
            $table->uuid('opportunity_id')->nullable();
            $table->string('type');
            $table->string('title');
            $table->text('description')->nullable();
            $table->uuid('performed_by');
            $table->dateTime('interaction_date')->useCurrent();
            $table->string('outcome')->nullable();
            $table->json('metadata')->nullable();
            $table->timestamps();
            $table->foreign('lead_id')->references('id')->on('leads')->nullOnDelete();
            $table->foreign('opportunity_id')->references('id')->on('opportunities')->nullOnDelete();
            $table->foreign('performed_by')->references('id')->on('users');
        });

        Schema::create('sales_funnels', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->string('name');
            $table->text('description')->nullable();
            $table->json('stages');
            $table->boolean('is_active')->default(true);
            $table->boolean('is_default')->default(false);
            $table->uuid('created_by');
            $table->uuid('branch_id')->nullable();
            $table->timestamps();
            $table->foreign('created_by')->references('id')->on('users');
            $table->foreign('branch_id')->references('id')->on('branches')->nullOnDelete();
            $table->index('is_active');
        });

        Schema::create('sales_targets', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->string('name');
            $table->text('description')->nullable();
            $table->string('type')->default('REVENUE'); // REVENUE, UNITS, DEALS, LEADS
            $table->float('target_value');
            $table->string('period')->default('MONTHLY');
            $table->dateTime('start_date');
            $table->dateTime('end_date');
            $table->string('assigned_to');
            $table->string('assigned_type')->default('USER'); // USER, TEAM, BRANCH
            $table->float('progress')->default(0);
            $table->string('status')->default('ACTIVE');
            $table->uuid('branch_id')->nullable();
            $table->json('metadata')->nullable();
            $table->timestamps();
            $table->index('assigned_to');
            $table->index('start_date');
        });

        Schema::create('lead_activities', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->uuid('lead_id');
            $table->string('type');
            $table->string('title');
            $table->text('description')->nullable();
            $table->uuid('performed_by');
            $table->json('metadata')->nullable();
            $table->timestamps();
            $table->foreign('lead_id')->references('id')->on('leads')->onDelete('cascade');
            $table->foreign('performed_by')->references('id')->on('users');
            $table->index(['lead_id', 'created_at']);
        });

        Schema::create('lead_communications', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->uuid('lead_id');
            $table->string('type'); // EMAIL, SMS, CALL, WHATSAPP
            $table->string('direction')->default('OUTBOUND');
            $table->string('subject')->nullable();
            $table->text('content');
            $table->string('status')->default('SENT');
            $table->dateTime('scheduled_at')->nullable();
            $table->dateTime('sent_at')->nullable();
            $table->dateTime('delivered_at')->nullable();
            $table->dateTime('opened_at')->nullable();
            $table->dateTime('clicked_at')->nullable();
            $table->dateTime('replied_at')->nullable();
            $table->json('metadata')->nullable();
            $table->timestamps();
            $table->index('lead_id');
            $table->index('status');
        });

        Schema::create('marketing_campaigns', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->string('name');
            $table->text('description')->nullable();
            $table->string('type'); // EMAIL, SMS, SOCIAL_MEDIA, ADVERTISEMENT, EVENT, REFERRAL, LOYALTY
            $table->string('category')->nullable();
            $table->string('status')->default('DRAFT');
            $table->dateTime('start_date');
            $table->dateTime('end_date')->nullable();
            $table->float('budget')->nullable();
            $table->json('target_audience')->nullable();
            $table->json('content')->nullable();
            $table->json('channels')->nullable();
            $table->json('goals')->nullable();
            $table->uuid('created_by');
            $table->uuid('approved_by')->nullable();
            $table->dateTime('approved_at')->nullable();
            $table->uuid('branch_id')->nullable();
            $table->json('tags')->nullable();
            $table->json('attachments')->nullable();
            $table->json('metadata')->nullable();
            $table->timestamps();
            $table->foreign('created_by')->references('id')->on('users');
            $table->foreign('branch_id')->references('id')->on('branches')->nullOnDelete();
            $table->index(['type', 'status']);
            $table->index('created_by');
        });

        Schema::create('campaign_members', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->uuid('campaign_id');
            $table->uuid('customer_id')->nullable();
            $table->uuid('lead_id')->nullable();
            $table->string('status')->default('ACTIVE');
            $table->json('metadata')->nullable();
            $table->timestamps();
            $table->foreign('campaign_id')->references('id')->on('marketing_campaigns')->onDelete('cascade');
            $table->foreign('customer_id')->references('id')->on('customer_profiles')->nullOnDelete();
            $table->foreign('lead_id')->references('id')->on('leads')->nullOnDelete();
        });

        Schema::create('marketing_automations', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->uuid('campaign_id');
            $table->string('name');
            $table->string('trigger_type');
            $table->json('trigger_config')->nullable();
            $table->string('action_type');
            $table->json('action_config')->nullable();
            $table->boolean('is_active')->default(true);
            $table->integer('delay_minutes')->default(0);
            $table->integer('executions_count')->default(0);
            $table->json('metadata')->nullable();
            $table->timestamps();
            $table->foreign('campaign_id')->references('id')->on('marketing_campaigns')->onDelete('cascade');
        });

        Schema::create('marketing_metrics', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->dateTime('date');
            $table->string('period')->default('DAILY');
            $table->integer('campaigns_sent')->default(0);
            $table->integer('emails_sent')->default(0);
            $table->integer('emails_opened')->default(0);
            $table->integer('emails_clicked')->default(0);
            $table->integer('sms_sent')->default(0);
            $table->integer('sms_delivered')->default(0);
            $table->integer('leads_generated')->default(0);
            $table->integer('leads_converted')->default(0);
            $table->float('conversion_rate')->default(0);
            $table->float('cost_per_lead')->default(0);
            $table->float('cost_per_acquisition')->default(0);
            $table->float('revenue_generated')->default(0);
            $table->float('roi')->default(0);
            $table->json('top_campaigns')->nullable();
            $table->uuid('branch_id')->nullable();
            $table->json('metadata')->nullable();
            $table->timestamps();
            $table->unique(['date', 'period', 'branch_id']);
            $table->index(['date', 'period']);
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('marketing_metrics');
        Schema::dropIfExists('marketing_automations');
        Schema::dropIfExists('campaign_members');
        Schema::dropIfExists('marketing_campaigns');
        Schema::dropIfExists('lead_communications');
        Schema::dropIfExists('lead_activities');
        Schema::dropIfExists('sales_targets');
        Schema::dropIfExists('sales_funnels');
        Schema::dropIfExists('crm_interactions');
        Schema::dropIfExists('customer_feedbacks');
        Schema::dropIfExists('customer_lifecycles');
        Schema::dropIfExists('customer_journeys');
        Schema::dropIfExists('customer_interactions');
    }
};
