<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        // B5: Support additions
        Schema::create('ticket_timelines', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->uuid('ticket_id');
            $table->string('action'); // CREATED, ASSIGNED, STATUS_CHANGED, COMMENT, ESCALATED, RESOLVED, CLOSED, REOPENED
            $table->text('description');
            $table->uuid('performed_by');
            $table->json('metadata')->nullable();
            $table->timestamps();
            $table->foreign('ticket_id')->references('id')->on('support_tickets')->onDelete('cascade');
            $table->foreign('performed_by')->references('id')->on('users');
            $table->index(['ticket_id', 'created_at']);
        });

        Schema::create('service_evaluations', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->uuid('customer_id');
            $table->string('service_type');
            $table->string('service_id')->nullable();
            $table->integer('overall_rating')->default(0);
            $table->integer('quality_rating')->nullable();
            $table->integer('speed_rating')->nullable();
            $table->integer('staff_rating')->nullable();
            $table->integer('value_rating')->nullable();
            $table->text('recommendations')->nullable();
            $table->boolean('would_recommend')->nullable();
            $table->text('feedback')->nullable();
            $table->string('status')->default('PENDING');
            $table->uuid('reviewed_by')->nullable();
            $table->dateTime('reviewed_at')->nullable();
            $table->text('response')->nullable();
            $table->json('tags')->nullable();
            $table->json('attachments')->nullable();
            $table->boolean('is_public')->default(false);
            $table->timestamps();
            $table->foreign('customer_id')->references('id')->on('users');
            $table->index('customer_id');
            $table->index('overall_rating');
            $table->index('status');
        });

        Schema::create('customer_service_metrics', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->dateTime('date');
            $table->string('period')->default('DAILY');
            $table->integer('tickets_received')->default(0);
            $table->integer('tickets_resolved')->default(0);
            $table->integer('tickets_open')->default(0);
            $table->float('avg_resolution_time')->default(0);
            $table->float('avg_response_time')->default(0);
            $table->float('customer_satisfaction')->default(0);
            $table->integer('escalations')->default(0);
            $table->integer('complaints')->default(0);
            $table->integer('complaints_resolved')->default(0);
            $table->integer('evaluations')->default(0);
            $table->float('avg_rating')->default(0);
            $table->uuid('branch_id')->nullable();
            $table->json('metadata')->nullable();
            $table->timestamps();
            $table->unique(['date', 'period', 'branch_id']);
        });

        Schema::create('knowledge_base_ratings', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->uuid('article_id');
            $table->uuid('user_id')->nullable();
            $table->integer('rating');
            $table->boolean('helpful');
            $table->text('feedback')->nullable();
            $table->string('ip_address')->nullable();
            $table->string('user_agent')->nullable();
            $table->timestamps();
            $table->foreign('article_id')->references('id')->on('knowledge_base_articles')->onDelete('cascade');
            $table->foreign('user_id')->references('id')->on('users')->nullOnDelete();
            $table->unique(['article_id', 'user_id']);
        });

        // B6: Insurance
        Schema::create('insurance_companies', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->string('name')->unique();
            $table->string('code')->unique();
            $table->text('description')->nullable();
            $table->string('contact_person')->nullable();
            $table->string('phone')->nullable();
            $table->string('email')->nullable();
            $table->string('address')->nullable();
            $table->string('website')->nullable();
            $table->boolean('is_active')->default(true);
            $table->uuid('branch_id')->nullable();
            $table->uuid('created_by');
            $table->timestamps();
            $table->foreign('created_by')->references('id')->on('users');
        });

        Schema::create('insurance_policies', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->string('policy_number')->unique();
            $table->uuid('vehicle_id');
            $table->uuid('customer_id');
            $table->uuid('company_id');
            $table->string('type'); // COMPREHENSIVE, THIRD_PARTY, COLLISION, LIABILITY
            $table->dateTime('start_date');
            $table->dateTime('end_date');
            $table->float('premium');
            $table->json('coverage')->nullable();
            $table->float('deductible')->default(0);
            $table->string('status')->default('PENDING');
            $table->json('documents')->nullable();
            $table->text('notes')->nullable();
            $table->uuid('created_by');
            $table->uuid('approved_by')->nullable();
            $table->dateTime('approved_at')->nullable();
            $table->timestamps();
            $table->foreign('vehicle_id')->references('id')->on('vehicles');
            $table->foreign('customer_id')->references('id')->on('users');
            $table->foreign('company_id')->references('id')->on('insurance_companies');
            $table->foreign('created_by')->references('id')->on('users');
            $table->index('status');
        });

        Schema::create('insurance_claims', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->string('claim_number')->unique();
            $table->uuid('policy_id');
            $table->uuid('vehicle_id');
            $table->uuid('customer_id');
            $table->string('type');
            $table->text('description');
            $table->dateTime('incident_date');
            $table->string('incident_location')->nullable();
            $table->float('estimated_amount')->nullable();
            $table->float('approved_amount')->nullable();
            $table->string('status')->default('PENDING');
            $table->json('documents')->nullable();
            $table->text('notes')->nullable();
            $table->uuid('assigned_to')->nullable();
            $table->dateTime('resolved_at')->nullable();
            $table->uuid('created_by');
            $table->timestamps();
            $table->foreign('policy_id')->references('id')->on('insurance_policies');
            $table->foreign('vehicle_id')->references('id')->on('vehicles');
            $table->foreign('customer_id')->references('id')->on('users');
            $table->foreign('created_by')->references('id')->on('users');
            $table->index('status');
        });

        Schema::create('insurance_payments', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->uuid('policy_id')->nullable();
            $table->uuid('claim_id')->nullable();
            $table->string('type');
            $table->float('amount');
            $table->string('currency')->default('EGP');
            $table->string('payment_method');
            $table->string('reference')->nullable();
            $table->string('status')->default('PENDING');
            $table->dateTime('payment_date')->nullable();
            $table->dateTime('due_date')->nullable();
            $table->text('notes')->nullable();
            $table->uuid('created_by');
            $table->uuid('approved_by')->nullable();
            $table->timestamps();
            $table->foreign('policy_id')->references('id')->on('insurance_policies')->nullOnDelete();
            $table->foreign('claim_id')->references('id')->on('insurance_claims')->nullOnDelete();
            $table->foreign('created_by')->references('id')->on('users');
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('insurance_payments');
        Schema::dropIfExists('insurance_claims');
        Schema::dropIfExists('insurance_policies');
        Schema::dropIfExists('insurance_companies');
        Schema::dropIfExists('knowledge_base_ratings');
        Schema::dropIfExists('customer_service_metrics');
        Schema::dropIfExists('service_evaluations');
        Schema::dropIfExists('ticket_timelines');
    }
};
