<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        // B7: Maintenance
        Schema::create('maintenance_schedules', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->uuid('vehicle_id');
            $table->string('type'); // OIL_CHANGE, TIRE_ROTATION, BRAKE_INSPECTION, etc.
            $table->string('title');
            $table->text('description')->nullable();
            $table->integer('interval'); // days
            $table->integer('interval_km')->nullable();
            $table->dateTime('last_service')->nullable();
            $table->dateTime('next_service');
            $table->float('estimated_cost')->nullable();
            $table->string('priority')->default('PENDING');
            $table->boolean('is_active')->default(true);
            $table->uuid('created_by');
            $table->timestamps();
            $table->foreign('vehicle_id')->references('id')->on('vehicles');
            $table->foreign('created_by')->references('id')->on('users');
            $table->index('vehicle_id');
            $table->index('next_service');
        });

        Schema::create('maintenance_records', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->uuid('vehicle_id');
            $table->uuid('schedule_id')->nullable();
            $table->string('type');
            $table->string('title');
            $table->text('description');
            $table->float('cost');
            $table->string('technician');
            $table->dateTime('start_date');
            $table->dateTime('end_date')->nullable();
            $table->string('status')->default('IN_PROGRESS');
            $table->text('notes')->nullable();
            $table->json('parts')->nullable();
            $table->float('labor_hours')->nullable();
            $table->integer('odometer')->nullable();
            $table->uuid('created_by');
            $table->timestamps();
            $table->foreign('vehicle_id')->references('id')->on('vehicles');
            $table->foreign('schedule_id')->references('id')->on('maintenance_schedules')->nullOnDelete();
            $table->foreign('created_by')->references('id')->on('users');
            $table->index('vehicle_id');
            $table->index('status');
        });

        Schema::create('maintenance_reminders', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->uuid('schedule_id');
            $table->uuid('vehicle_id');
            $table->string('title');
            $table->text('message');
            $table->dateTime('reminder_date');
            $table->dateTime('sent_date')->nullable();
            $table->string('status')->default('PENDING');
            $table->string('type')->default('EMAIL');
            $table->uuid('created_by');
            $table->timestamps();
            $table->foreign('schedule_id')->references('id')->on('maintenance_schedules');
            $table->foreign('vehicle_id')->references('id')->on('vehicles');
            $table->foreign('created_by')->references('id')->on('users');
            $table->index('reminder_date');
        });

        Schema::create('maintenance_parts', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->string('part_number')->unique();
            $table->string('name');
            $table->string('category'); // ENGINE, BRAKES, SUSPENSION, ELECTRICAL, BODY, etc.
            $table->text('description')->nullable();
            $table->float('cost');
            $table->float('price');
            $table->integer('quantity')->default(0);
            $table->integer('min_stock')->default(0);
            $table->integer('max_stock')->nullable();
            $table->string('location')->nullable();
            $table->string('supplier')->nullable();
            $table->string('status')->default('AVAILABLE');
            $table->string('barcode')->nullable();
            $table->string('image_url')->nullable();
            $table->uuid('created_by');
            $table->timestamps();
            $table->foreign('created_by')->references('id')->on('users');
            $table->index('part_number');
            $table->index('category');
            $table->index('status');
        });

        // B8: Commerce
        Schema::create('products', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->string('name');
            $table->text('description');
            $table->float('price');
            $table->string('category');
            $table->json('images');
            $table->boolean('in_stock')->default(true);
            $table->integer('quantity')->default(0);
            $table->boolean('featured')->default(false);
            $table->string('status')->default('ACTIVE');
            $table->uuid('created_by');
            $table->timestamps();
            $table->foreign('created_by')->references('id')->on('users');
            $table->index('name');
            $table->index('category');
            $table->index('status');
        });

        Schema::create('orders', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->string('order_number')->unique();
            $table->uuid('customer_id');
            $table->string('status')->default('PENDING');
            $table->float('subtotal');
            $table->float('tax_amount')->default(0);
            $table->float('shipping_amount')->default(0);
            $table->float('total');
            $table->json('shipping_address');
            $table->json('billing_address')->nullable();
            $table->string('payment_method')->nullable();
            $table->string('payment_status')->default('PENDING');
            $table->text('notes')->nullable();
            $table->uuid('created_by');
            $table->timestamps();
            $table->foreign('customer_id')->references('id')->on('users');
            $table->foreign('created_by')->references('id')->on('users');
            $table->index('order_number');
            $table->index('status');
        });

        Schema::create('order_items', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->uuid('order_id');
            $table->uuid('product_id');
            $table->integer('quantity');
            $table->float('price');
            $table->float('total_price');
            $table->timestamps();
            $table->foreign('order_id')->references('id')->on('orders')->onDelete('cascade');
            $table->foreign('product_id')->references('id')->on('products');
        });

        Schema::create('order_payments', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->uuid('order_id');
            $table->float('amount');
            $table->string('currency')->default('EGP');
            $table->string('status')->default('PENDING');
            $table->string('payment_method');
            $table->string('transaction_id')->nullable();
            $table->string('receipt_url')->nullable();
            $table->text('notes')->nullable();
            $table->timestamps();
            $table->foreign('order_id')->references('id')->on('orders');
            $table->index('status');
        });

        Schema::create('promotions', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->string('title');
            $table->text('description');
            $table->string('type'); // PERCENTAGE, FIXED, FREE_SHIPPING
            $table->float('value');
            $table->string('code')->unique();
            $table->dateTime('start_date');
            $table->dateTime('end_date');
            $table->integer('usage_limit');
            $table->integer('used_count')->default(0);
            $table->boolean('active')->default(true);
            $table->uuid('created_by');
            $table->timestamps();
            $table->foreign('created_by')->references('id')->on('users');
            $table->index('code');
            $table->index('active');
        });

        Schema::create('promotion_usages', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->uuid('promotion_id');
            $table->uuid('order_id');
            $table->float('discount');
            $table->dateTime('used_at')->useCurrent();
            $table->foreign('promotion_id')->references('id')->on('promotions');
            $table->foreign('order_id')->references('id')->on('orders');
        });

        Schema::create('product_reviews', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->uuid('product_id');
            $table->uuid('customer_id')->nullable();
            $table->integer('rating');
            $table->string('title')->nullable();
            $table->text('review');
            $table->json('images')->nullable();
            $table->string('status')->default('PENDING');
            $table->uuid('approved_by')->nullable();
            $table->dateTime('approved_at')->nullable();
            $table->boolean('is_anonymous')->default(false);
            $table->integer('helpful')->default(0);
            $table->timestamps();
            $table->foreign('product_id')->references('id')->on('products');
            $table->foreign('customer_id')->references('id')->on('users')->nullOnDelete();
            $table->index('rating');
            $table->index('status');
        });

        Schema::create('commerce_settings', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->json('settings');
            $table->timestamps();
        });

        // B9: Contracts
        Schema::create('contracts', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->string('contract_number')->unique();
            $table->string('type'); // SALE, LEASE, RENTAL, SERVICE, MAINTENANCE, INSURANCE
            $table->uuid('customer_id');
            $table->uuid('vehicle_id')->nullable();
            $table->dateTime('start_date');
            $table->dateTime('end_date')->nullable();
            $table->float('value');
            $table->string('status')->default('DRAFT');
            $table->json('terms')->nullable();
            $table->json('attachments')->nullable();
            $table->uuid('created_by');
            $table->uuid('approved_by')->nullable();
            $table->dateTime('approved_at')->nullable();
            $table->uuid('branch_id')->nullable();
            $table->timestamps();
            $table->foreign('customer_id')->references('id')->on('users');
            $table->foreign('vehicle_id')->references('id')->on('vehicles')->nullOnDelete();
            $table->foreign('created_by')->references('id')->on('users');
            $table->foreign('branch_id')->references('id')->on('branches')->nullOnDelete();
            $table->index(['type', 'status']);
        });

        Schema::create('warranties', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->string('warranty_number')->unique();
            $table->uuid('contract_id');
            $table->uuid('vehicle_id');
            $table->string('type'); // BASIC, EXTENDED, POWERTRAIN, COMPREHENSIVE
            $table->dateTime('start_date');
            $table->dateTime('end_date');
            $table->json('coverage')->nullable();
            $table->text('terms')->nullable();
            $table->string('status')->default('ACTIVE');
            $table->timestamps();
            $table->foreign('contract_id')->references('id')->on('contracts');
            $table->foreign('vehicle_id')->references('id')->on('vehicles');
            $table->index('vehicle_id');
        });

        Schema::create('warranty_claims', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->string('claim_number')->unique();
            $table->uuid('warranty_id');
            $table->uuid('vehicle_id');
            $table->uuid('customer_id');
            $table->text('description');
            $table->dateTime('claim_date');
            $table->string('status')->default('PENDING');
            $table->float('estimated_cost')->nullable();
            $table->float('actual_cost')->nullable();
            $table->uuid('approved_by')->nullable();
            $table->dateTime('approved_at')->nullable();
            $table->dateTime('resolved_at')->nullable();
            $table->text('resolution')->nullable();
            $table->json('attachments')->nullable();
            $table->uuid('created_by');
            $table->timestamps();
            $table->foreign('warranty_id')->references('id')->on('warranties');
            $table->foreign('vehicle_id')->references('id')->on('vehicles');
            $table->foreign('customer_id')->references('id')->on('users');
            $table->foreign('created_by')->references('id')->on('users');
            $table->index('status');
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('warranty_claims');
        Schema::dropIfExists('warranties');
        Schema::dropIfExists('contracts');
        Schema::dropIfExists('commerce_settings');
        Schema::dropIfExists('product_reviews');
        Schema::dropIfExists('promotion_usages');
        Schema::dropIfExists('promotions');
        Schema::dropIfExists('order_payments');
        Schema::dropIfExists('order_items');
        Schema::dropIfExists('orders');
        Schema::dropIfExists('products');
        Schema::dropIfExists('maintenance_parts');
        Schema::dropIfExists('maintenance_reminders');
        Schema::dropIfExists('maintenance_records');
        Schema::dropIfExists('maintenance_schedules');
    }
};
