<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        // System & Logging
        if (! Schema::hasTable('security_logs')) {
            Schema::create('security_logs', function (Blueprint $table) {
                $table->uuid('id')->primary();
                $table->uuid('user_id')->nullable();
                $table->string('action');
                $table->string('severity')->default('INFO');
                $table->text('description')->nullable();
                $table->string('ip_address')->nullable();
                $table->string('user_agent')->nullable();
                $table->json('metadata')->nullable();
                $table->timestamps();
                $table->foreign('user_id')->references('id')->on('users')->nullOnDelete();
                $table->index('user_id');
                $table->index('action');
                $table->index('created_at');
            });
        }

        if (! Schema::hasTable('activity_logs')) {
            Schema::create('activity_logs', function (Blueprint $table) {
                $table->uuid('id')->primary();
                $table->uuid('user_id')->nullable();
                $table->string('entity_type');
                $table->string('entity_id');
                $table->string('action');
                $table->json('old_values')->nullable();
                $table->json('new_values')->nullable();
                $table->string('ip_address')->nullable();
                $table->timestamps();
                $table->foreign('user_id')->references('id')->on('users')->nullOnDelete();
                $table->index(['entity_type', 'entity_id']);
                $table->index('user_id');
            });
        }

        if (! Schema::hasTable('system_notifications')) {
            Schema::create('system_notifications', function (Blueprint $table) {
                $table->uuid('id')->primary();
                $table->uuid('user_id');
                $table->string('type');
                $table->string('title');
                $table->text('message');
                $table->string('channel')->default('EMAIL');
                $table->string('priority')->default('MEDIUM');
                $table->string('status')->default('PENDING');
                $table->dateTime('sent_at')->nullable();
                $table->dateTime('read_at')->nullable();
                $table->string('link_url')->nullable();
                $table->json('metadata')->nullable();
                $table->timestamps();
                $table->foreign('user_id')->references('id')->on('users');
                $table->index(['user_id', 'status']);
            });
        }

        if (! Schema::hasTable('email_templates')) {
            Schema::create('email_templates', function (Blueprint $table) {
                $table->uuid('id')->primary();
                $table->string('name');
                $table->string('type');
                $table->string('subject');
                $table->text('body');
                $table->json('variables')->nullable();
                $table->boolean('is_active')->default(true);
                $table->timestamps();
                $table->unique(['name', 'type']);
            });
        }

        if (! Schema::hasTable('tasks')) {
            Schema::create('tasks', function (Blueprint $table) {
                $table->uuid('id')->primary();
                $table->string('title');
                $table->text('description')->nullable();
                $table->uuid('assigned_to');
                $table->uuid('assigned_by');
                $table->string('priority')->default('MEDIUM');
                $table->string('status')->default('PENDING');
                $table->dateTime('due_date')->nullable();
                $table->dateTime('completed_at')->nullable();
                $table->json('tags')->nullable();
                $table->uuid('branch_id')->nullable();
                $table->timestamps();
                $table->foreign('assigned_to')->references('id')->on('users');
                $table->foreign('assigned_by')->references('id')->on('users');
                $table->foreign('branch_id')->references('id')->on('branches')->nullOnDelete();
                $table->index(['assigned_to', 'status']);
            });
        }

        if (! Schema::hasTable('task_comments')) {
            Schema::create('task_comments', function (Blueprint $table) {
                $table->uuid('id')->primary();
                $table->uuid('task_id');
                $table->uuid('author_id');
                $table->text('content');
                $table->timestamps();
                $table->foreign('task_id')->references('id')->on('tasks')->onDelete('cascade');
                $table->foreign('author_id')->references('id')->on('users');
            });
        }

        if (! Schema::hasTable('calendar_events')) {
            Schema::create('calendar_events', function (Blueprint $table) {
                $table->uuid('id')->primary();
                $table->string('title');
                $table->text('description')->nullable();
                $table->string('type')->default('APPOINTMENT');
                $table->dateTime('start_time');
                $table->dateTime('end_time');
                $table->boolean('all_day')->default(false);
                $table->string('location')->nullable();
                $table->string('status')->default('SCHEDULED');
                $table->string('color')->nullable();
                $table->uuid('created_by');
                $table->uuid('assigned_to')->nullable();
                $table->uuid('branch_id')->nullable();
                $table->json('metadata')->nullable();
                $table->timestamps();
                $table->foreign('created_by')->references('id')->on('users');
                $table->foreign('assigned_to')->references('id')->on('users')->nullOnDelete();
                $table->foreign('branch_id')->references('id')->on('branches')->nullOnDelete();
                $table->index('start_time');
            });
        }

        if (! Schema::hasTable('holidays')) {
            Schema::create('holidays', function (Blueprint $table) {
                $table->uuid('id')->primary();
                $table->string('name');
                $table->date('date');
                $table->boolean('is_recurring')->default(false);
                $table->text('description')->nullable();
                $table->boolean('is_active')->default(true);
                $table->timestamps();
                $table->unique(['name', 'date']);
            });
        }

        if (! Schema::hasTable('time_slots')) {
            Schema::create('time_slots', function (Blueprint $table) {
                $table->uuid('id')->primary();
                $table->string('day_of_week');
                $table->time('start_time');
                $table->time('end_time');
                $table->boolean('is_available')->default(true);
                $table->uuid('branch_id')->nullable();
                $table->string('service_type')->nullable();
                $table->integer('max_bookings')->default(1);
                $table->timestamps();
                $table->foreign('branch_id')->references('id')->on('branches')->nullOnDelete();
            });
        }

        if (! Schema::hasTable('media')) {
            Schema::create('media', function (Blueprint $table) {
                $table->uuid('id')->primary();
                $table->string('filename');
                $table->string('original_name');
                $table->string('path');
                $table->string('url');
                $table->string('thumbnail_url')->nullable();
                $table->string('mime_type');
                $table->integer('size');
                $table->integer('width')->nullable();
                $table->integer('height')->nullable();
                $table->string('alt_text')->nullable();
                $table->string('title')->nullable();
                $table->text('description')->nullable();
                $table->string('tags')->nullable();
                $table->string('category')->default('other');
                $table->string('entity_id')->nullable();
                $table->boolean('is_public')->default(true);
                $table->boolean('is_featured')->default(false);
                $table->integer('order')->default(0);
                $table->text('metadata')->nullable();
                $table->uuid('created_by');
                $table->timestamps();
                $table->foreign('created_by')->references('id')->on('users');
            });
        }

        if (! Schema::hasTable('popup_configs')) {
            Schema::create('popup_configs', function (Blueprint $table) {
                $table->uuid('id')->primary();
                $table->string('title')->nullable();
                $table->text('content')->nullable();
                $table->string('image_url')->nullable();
                $table->string('link_url')->nullable();
                $table->string('button_text')->nullable();
                $table->string('button_color')->default('#3b82f6');
                $table->string('text_color')->default('#ffffff');
                $table->string('background_color')->default('#1f2937');
                $table->string('position')->default('BOTTOM_RIGHT');
                $table->integer('show_delay')->default(3000);
                $table->boolean('auto_hide')->default(true);
                $table->integer('hide_delay')->default(10000);
                $table->boolean('is_active')->default(true);
                $table->string('show_on_pages')->default('homepage');
                $table->string('target_audience')->default('all');
                $table->dateTime('start_date')->nullable();
                $table->dateTime('end_date')->nullable();
                $table->integer('priority')->default(0);
                $table->timestamps();
            });
        }

        // CMS Content tables
        if (! Schema::hasTable('header_contents')) {
            Schema::create('header_contents', function (Blueprint $table) {
                $table->uuid('id')->primary();
                $table->string('logo_url')->nullable();
                $table->string('logo_alt')->nullable();
                $table->string('phone')->nullable();
                $table->string('email')->nullable();
                $table->text('announcement')->nullable();
                $table->boolean('show_announcement')->default(false);
                $table->json('settings')->nullable();
                $table->timestamps();
            });
        }

        if (! Schema::hasTable('header_navigations')) {
            Schema::create('header_navigations', function (Blueprint $table) {
                $table->uuid('id')->primary();
                $table->string('label');
                $table->string('url');
                $table->integer('order')->default(0);
                $table->boolean('is_active')->default(true);
                $table->uuid('parent_id')->nullable();
                $table->timestamps();
            });
        }

        if (! Schema::hasTable('footer_contents')) {
            Schema::create('footer_contents', function (Blueprint $table) {
                $table->uuid('id')->primary();
                $table->text('description')->nullable();
                $table->string('copyright')->nullable();
                $table->json('settings')->nullable();
                $table->timestamps();
            });
        }

        if (! Schema::hasTable('footer_columns')) {
            Schema::create('footer_columns', function (Blueprint $table) {
                $table->uuid('id')->primary();
                $table->string('title');
                $table->string('type')->default('LINKS'); // LINKS, TEXT, CONTACT, SOCIAL
                $table->json('items')->nullable();
                $table->integer('order')->default(0);
                $table->boolean('is_active')->default(true);
                $table->timestamps();
            });
        }

        if (! Schema::hasTable('footer_socials')) {
            Schema::create('footer_socials', function (Blueprint $table) {
                $table->uuid('id')->primary();
                $table->string('platform');
                $table->string('url');
                $table->string('icon')->nullable();
                $table->integer('order')->default(0);
                $table->boolean('is_active')->default(true);
                $table->timestamps();
            });
        }

        if (! Schema::hasTable('homepage_settings')) {
            Schema::create('homepage_settings', function (Blueprint $table) {
                $table->uuid('id')->primary();
                $table->string('section');
                $table->json('content')->nullable();
                $table->boolean('is_visible')->default(true);
                $table->integer('order')->default(0);
                $table->timestamps();
                $table->unique('section');
            });
        }

        if (! Schema::hasTable('company_info')) {
            Schema::create('company_info', function (Blueprint $table) {
                $table->uuid('id')->primary();
                $table->string('name');
                $table->text('description')->nullable();
                $table->text('mission')->nullable();
                $table->text('vision')->nullable();
                $table->string('logo_url')->nullable();
                $table->json('settings')->nullable();
                $table->timestamps();
            });
        }

        if (! Schema::hasTable('company_values')) {
            Schema::create('company_values', function (Blueprint $table) {
                $table->uuid('id')->primary();
                $table->string('title');
                $table->text('description')->nullable();
                $table->string('icon')->nullable();
                $table->integer('order')->default(0);
                $table->timestamps();
            });
        }

        if (! Schema::hasTable('company_stats')) {
            Schema::create('company_stats', function (Blueprint $table) {
                $table->uuid('id')->primary();
                $table->string('label');
                $table->string('value');
                $table->string('icon')->nullable();
                $table->integer('order')->default(0);
                $table->timestamps();
            });
        }

        if (! Schema::hasTable('company_features')) {
            Schema::create('company_features', function (Blueprint $table) {
                $table->uuid('id')->primary();
                $table->string('title');
                $table->text('description')->nullable();
                $table->string('icon')->nullable();
                $table->string('image_url')->nullable();
                $table->integer('order')->default(0);
                $table->timestamps();
            });
        }

        if (! Schema::hasTable('contact_info')) {
            Schema::create('contact_info', function (Blueprint $table) {
                $table->uuid('id')->primary();
                $table->string('phone')->nullable();
                $table->string('email')->nullable();
                $table->string('address')->nullable();
                $table->string('map_url')->nullable();
                $table->json('working_hours')->nullable();
                $table->json('social_links')->nullable();
                $table->timestamps();
            });
        }

        if (! Schema::hasTable('timeline_events')) {
            Schema::create('timeline_events', function (Blueprint $table) {
                $table->uuid('id')->primary();
                $table->string('title');
                $table->text('description')->nullable();
                $table->date('event_date');
                $table->string('icon')->nullable();
                $table->string('color')->nullable();
                $table->integer('order')->default(0);
                $table->boolean('is_active')->default(true);
                $table->timestamps();
            });
        }

        if (! Schema::hasTable('service_items')) {
            Schema::create('service_items', function (Blueprint $table) {
                $table->uuid('id')->primary();
                $table->string('name');
                $table->text('description')->nullable();
                $table->string('icon')->nullable();
                $table->string('image_url')->nullable();
                $table->float('price')->nullable();
                $table->string('category')->nullable();
                $table->integer('order')->default(0);
                $table->boolean('is_active')->default(true);
                $table->timestamps();
            });
        }

        // Submission tables
        if (! Schema::hasTable('service_booking_submissions')) {
            Schema::create('service_booking_submissions', function (Blueprint $table) {
                $table->uuid('id')->primary();
                $table->string('name');
                $table->string('email');
                $table->string('phone');
                $table->string('vehicle_type');
                $table->string('service_type');
                $table->dateTime('preferred_date');
                $table->string('preferred_time');
                $table->text('message')->nullable();
                $table->string('status')->default('PENDING');
                $table->dateTime('submitted_at')->useCurrent();
                $table->timestamps();
            });
        }

        if (! Schema::hasTable('test_drive_submissions')) {
            Schema::create('test_drive_submissions', function (Blueprint $table) {
                $table->uuid('id')->primary();
                $table->string('name');
                $table->string('email');
                $table->string('phone');
                $table->uuid('vehicle_id');
                $table->string('vehicle_model');
                $table->dateTime('preferred_date');
                $table->string('preferred_time');
                $table->text('message')->nullable();
                $table->string('status')->default('PENDING');
                $table->dateTime('submitted_at')->useCurrent();
                $table->timestamps();
            });
        }

        if (! Schema::hasTable('consultation_submissions')) {
            Schema::create('consultation_submissions', function (Blueprint $table) {
                $table->uuid('id')->primary();
                $table->string('name');
                $table->string('email');
                $table->string('phone');
                $table->string('consultation_type');
                $table->dateTime('preferred_date');
                $table->string('preferred_time');
                $table->text('message')->nullable();
                $table->string('status')->default('PENDING');
                $table->dateTime('submitted_at')->useCurrent();
                $table->timestamps();
            });
        }

        // Branch extensions
        if (! Schema::hasTable('branch_permissions')) {
            Schema::create('branch_permissions', function (Blueprint $table) {
                $table->uuid('id')->primary();
                $table->uuid('branch_id');
                $table->string('permission');
                $table->boolean('granted')->default(true);
                $table->timestamps();
                $table->foreign('branch_id')->references('id')->on('branches')->onDelete('cascade');
                $table->unique(['branch_id', 'permission']);
            });
        }

        if (! Schema::hasTable('branch_transfers')) {
            Schema::create('branch_transfers', function (Blueprint $table) {
                $table->uuid('id')->primary();
                $table->uuid('from_branch_id');
                $table->uuid('to_branch_id');
                $table->string('entity_type'); // VEHICLE, INVENTORY, EMPLOYEE
                $table->string('entity_id');
                $table->string('status')->default('PENDING');
                $table->text('reason')->nullable();
                $table->uuid('requested_by');
                $table->uuid('approved_by')->nullable();
                $table->dateTime('approved_at')->nullable();
                $table->timestamps();
                $table->foreign('from_branch_id')->references('id')->on('branches');
                $table->foreign('to_branch_id')->references('id')->on('branches');
                $table->foreign('requested_by')->references('id')->on('users');
            });
        }

        if (! Schema::hasTable('branch_budgets')) {
            Schema::create('branch_budgets', function (Blueprint $table) {
                $table->uuid('id')->primary();
                $table->uuid('branch_id');
                $table->string('period'); // e.g. '2026-Q1'
                $table->float('allocated_budget')->default(0);
                $table->float('spent_amount')->default(0);
                $table->string('status')->default('ACTIVE');
                $table->json('categories')->nullable();
                $table->text('notes')->nullable();
                $table->timestamps();
                $table->foreign('branch_id')->references('id')->on('branches');
                $table->unique(['branch_id', 'period']);
            });
        }

        // Stock alerts
        if (! Schema::hasTable('stock_alerts')) {
            Schema::create('stock_alerts', function (Blueprint $table) {
                $table->uuid('id')->primary();
                $table->uuid('item_id');
                $table->string('alert_type')->default('LOW_STOCK');
                $table->integer('threshold');
                $table->integer('current_quantity');
                $table->string('status')->default('ACTIVE');
                $table->dateTime('acknowledged_at')->nullable();
                $table->uuid('acknowledged_by')->nullable();
                $table->timestamps();
                $table->foreign('item_id')->references('id')->on('inventory_items');
                $table->index('status');
            });
        }
    }

    public function down(): void
    {
        // ... (Down method omitted for brevity, logic remains same)
        Schema::dropIfExists('stock_alerts');
        Schema::dropIfExists('branch_budgets');
        Schema::dropIfExists('branch_transfers');
        Schema::dropIfExists('branch_permissions');
        Schema::dropIfExists('consultation_submissions');
        Schema::dropIfExists('test_drive_submissions');
        Schema::dropIfExists('service_booking_submissions');
        Schema::dropIfExists('service_items');
        Schema::dropIfExists('timeline_events');
        Schema::dropIfExists('contact_info');
        Schema::dropIfExists('company_features');
        Schema::dropIfExists('company_stats');
        Schema::dropIfExists('company_values');
        Schema::dropIfExists('company_info');
        Schema::dropIfExists('homepage_settings');
        Schema::dropIfExists('footer_socials');
        Schema::dropIfExists('footer_columns');
        Schema::dropIfExists('footer_contents');
        Schema::dropIfExists('header_navigations');
        Schema::dropIfExists('header_contents');
        Schema::dropIfExists('popup_configs');
        Schema::dropIfExists('media');
        Schema::dropIfExists('time_slots');
        Schema::dropIfExists('holidays');
        Schema::dropIfExists('calendar_events');
        Schema::dropIfExists('task_comments');
        Schema::dropIfExists('tasks');
        Schema::dropIfExists('email_templates');
        Schema::dropIfExists('system_notifications');
        Schema::dropIfExists('activity_logs');
        Schema::dropIfExists('security_logs');
    }
};
