<?php

namespace Database\Seeders;

use App\Models\Invoice;
use App\Models\InvoiceItem;
use App\Models\ServiceBooking;
use App\Models\User;
use Illuminate\Database\Seeder;
use Illuminate\Support\Str;

class DemoDataSeeder extends Seeder
{
    public function run(): void
    {
        $customers = User::where('role', 'CUSTOMER')->get();
        $branch = \App\Models\Branch::first();
        $serviceTypes = \App\Models\ServiceType::all();

        if ($customers->isEmpty() || $serviceTypes->isEmpty()) {
            $this->command->warn('Please run OrganizationSeeder, UsersSeeder, and ServiceTypesSeeder first.');

            return;
        }

        // Service Bookings
        foreach ($customers as $customer) {
            try {
                foreach ($serviceTypes->random(rand(1, 3)) as $st) {
                    ServiceBooking::firstOrCreate(
                        ['customer_id' => $customer->id, 'service_type_id' => $st->id],
                        [
                            'id' => (string) Str::uuid(),
                            'customer_id' => $customer->id,
                            'service_type_id' => $st->id,
                            'date' => now()->subDays(rand(1, 60)),
                            'time' => ['09:00', '10:00', '11:00', '14:00', '15:00'][rand(0, 4)],
                            'status' => ['PENDING', 'CONFIRMED', 'COMPLETED', 'CANCELLED'][rand(0, 3)],
                            'notes' => 'حجز تجريبي',
                        ]
                    );
                }
            } catch (\Exception $e) {
                // Continue
            }
        }

        // Invoices
        $statuses = ['DRAFT', 'SENT', 'PAID', 'OVERDUE', 'PAID', 'PAID'];
        foreach ($customers as $customer) {
            for ($i = 0; $i < rand(2, 4); $i++) {
                try {
                    $status = $statuses[array_rand($statuses)];
                    $subtotal = rand(200, 5000);
                    $tax = round($subtotal * 0.14, 2);
                    $total = $subtotal + $tax;

                    $invoice = Invoice::create([
                        'id' => (string) Str::uuid(),
                        'invoice_number' => 'INV-'.strtoupper(Str::random(8)),
                        'customer_id' => $customer->id,
                        'branch_id' => $branch?->id,
                        'type' => 'SERVICE',
                        'status' => $status,
                        'payment_status' => $status === 'PAID' ? 'PAID' : 'PENDING',
                        'issue_date' => now()->subDays(rand(1, 90)),
                        'due_date' => now()->addDays(rand(1, 30)),
                        'subtotal' => $subtotal,
                        'tax_amount' => $tax,
                        'total_amount' => $total,
                        'paid_amount' => $status === 'PAID' ? $total : 0,
                        'currency' => 'EGP',
                        'created_by_id' => \App\Models\User::where('role', 'SUPER_ADMIN')->first()->id ?? \App\Models\User::first()->id,
                    ]);

                    // Invoice Items
                    for ($j = 0; $j < rand(1, 3); $j++) {
                        $qty = rand(1, 5);
                        $price = rand(50, 1500);
                        InvoiceItem::create([
                            'id' => (string) Str::uuid(),
                            'invoice_id' => $invoice->id,
                            'description' => ['خدمة صيانة', 'قطع غيار', 'فحص', 'غسيل', 'تلميع'][rand(0, 4)],
                            'quantity' => $qty,
                            'unit_price' => $price,
                            'total_price' => $qty * $price,
                            'tax_rate' => 14,
                            'tax_amount' => round($qty * $price * 0.14, 2),
                        ]);
                    }
                } catch (\Exception $e) {
                    // Continue
                }
            }
        }

        // Tickets
        $priorities = ['LOW', 'MEDIUM', 'HIGH', 'URGENT'];
        foreach ($customers->take(3) as $customer) {
            for ($i = 0; $i < rand(1, 3); $i++) {
                try {
                    \App\Models\Ticket::create([
                        'id' => (string) Str::uuid(),
                        'ticket_number' => 'TKT-'.strtoupper(Str::random(6)),
                        'customer_id' => $customer->id,
                        'subject' => ['مشكلة في الفاتورة', 'استفسار عن الحجز', 'شكوى خدمة', 'طلب استرجاع'][rand(0, 3)],
                        'description' => 'تم إنشاء هذه التذكرة كبيانات تجريبية للنظام.',
                        'status' => ['OPEN', 'IN_PROGRESS', 'RESOLVED', 'CLOSED'][rand(0, 3)],
                        'priority' => $priorities[rand(0, 3)],
                    ]);
                } catch (\Exception $e) {
                    // Continue
                }
            }
        }

        $this->command->info('Demo data seeded successfully!');
    }
}
