<?php

namespace Database\Seeders;

use App\Models\Department;
use App\Models\Employee;
use App\Models\LeaveRequest;
use App\Models\PayrollRecord;
use App\Models\Position;
use App\Models\User;
use Illuminate\Database\Seeder;
use Illuminate\Support\Str;

class EmployeeDataSeeder extends Seeder
{
    public function run(): void
    {
        $employees = User::whereIn('role', ['ADMIN', 'SALES_MANAGER', 'ACCOUNTANT', 'EMPLOYEE'])->get();
        $departments = Department::all();
        $positions = Position::all();
        $branch = \App\Models\Branch::first();

        if ($employees->isEmpty() || $departments->isEmpty()) {
            $this->command->warn('Please run OrganizationSeeder and UsersSeeder first.');

            return;
        }

        foreach ($employees as $emp) {
            // Create Employee record (employees table, not employee_profiles)
            $employee = Employee::firstOrCreate(['user_id' => $emp->id], [
                'employee_number' => 'EMP-'.strtoupper(Str::random(4)),
                'user_id' => $emp->id,
                'department_id' => $departments->random()->id,
                'position_id' => $positions->isNotEmpty() ? $positions->random()->id : null,
                'branch_id' => $branch?->id,
                'hire_date' => now()->subMonths(rand(6, 36)),
                'salary' => rand(5, 25) * 1000,
                'status' => 'ACTIVE',
            ]);

            // Leave Requests (user_id based)
            for ($i = 0; $i < rand(2, 3); $i++) {
                $start = now()->subDays(rand(10, 180));
                $days = rand(1, 5);
                LeaveRequest::create([
                    'user_id' => $emp->id,
                    'leave_type' => ['ANNUAL', 'SICK', 'PERSONAL', 'EMERGENCY'][rand(0, 3)],
                    'start_date' => $start,
                    'end_date' => $start->copy()->addDays($days),
                    'total_days' => $days,
                    'reason' => ['إجازة سنوية', 'إجازة مرضية', 'ظروف شخصية', 'حالة طوارئ'][rand(0, 3)],
                    'status' => ['PENDING', 'APPROVED', 'APPROVED', 'REJECTED'][rand(0, 3)],
                ]);
            }

            // Payroll Records (user_id based)
            for ($m = 0; $m < 3; $m++) {
                $baseSalary = $employee->salary ?? rand(5, 15) * 1000;
                $deductions = rand(200, 1500);
                $bonus = rand(0, 1) ? rand(500, 2000) : 0;
                PayrollRecord::create([
                    'user_id' => $emp->id,
                    'month' => now()->subMonths($m)->month,
                    'year' => now()->subMonths($m)->year,
                    'basic_salary' => $baseSalary,
                    'deductions' => $deductions,
                    'bonus' => $bonus,
                    'net_salary' => $baseSalary + $bonus - $deductions,
                    'status' => $m === 0 ? 'PENDING' : 'PAID',
                    'payment_date' => $m === 0 ? null : now()->subMonths($m)->endOfMonth(),
                ]);
            }
        }

        $this->command->info('Employee data seeded successfully!');
    }
}
