<?php

namespace Database\Seeders;

use App\Models\Branch;
use App\Models\CompanyFeature;
use App\Models\CompanyStat;
use App\Models\CompanyValue;
use App\Models\ContactInfo;
use App\Models\Permission;
use App\Models\RoleTemplate;
use App\Models\ServiceItem;
use App\Models\ServiceType;
use App\Models\Slider;
use App\Models\TimelineEvent;
use App\Models\User;
use App\Models\Vehicle;
use App\Models\VehicleImage;
use Exception;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Str;

class LegacyDataSeeder extends Seeder
{
    public function run()
    {
        Log::info('🚀 Starting Legacy Data Import (Robust Mode)...');
        $this->command->info('🚀 Starting Legacy Data Import (Robust Mode)...');

        // 1. Site Settings & Contact
        $this->seedSiteSettings();
        $this->seedContactInfo();

        // 2. Company CMS Data
        $this->seedCompanyInfo();
        $this->seedCompanyStats();
        $this->seedCompanyValues();
        $this->seedTimeline();
        $this->seedServiceItems();
        $this->seedCompanyFeatures();
        $this->seedFaqs();
        $this->seedCompanyInfo();

        // 3. Service Types (Booking)
        $this->seedServiceTypes();

        // 4. Permissions & Roles
        $this->seedPermissionsAndRoles();

        $mainBranch = null;
        try {
            // 5. Branches
            $mainBranch = $this->seedBranches();
            Log::info('✓ Branches seeded');
        } catch (Exception $e) {
            Log::error('❌ Branches failed: '.$e->getMessage());
        }

        if ($mainBranch) {
            // 6. Users
            $this->seedUsers($mainBranch);

            // 7. Vehicles (Passenger + Commercial)
            $this->seedVehicles($mainBranch);
        } else {
            Log::error('Skipping Users and Vehicles because Main Branch failed');
        }

        // 8. Sliders
        $this->seedSliders();

        $this->command->info('✅ Import Completed (Check logs for individual errors).');
    }

    private function seedSiteSettings()
    {
        try {
            if (Schema::hasTable('site_settings')) {
                DB::table('site_settings')->updateOrInsert(
                    ['id' => 'default'],
                    [
                        'site_title' => 'شركة الحمد لاستيراد السيارات',
                        'site_description' => 'الموزع المعتمد لسيارات تاتا في مدن القناة',
                        'contact_email' => 'info@elhamdimport.online',
                        'contact_phone' => '+20 2 12345678',
                        'contact_address' => 'بورسعيد، مصر',
                        'working_hours' => 'السبت - الخميس: 9:00 ص - 5:00 م، الجمعة: مغلق',
                        'primary_color' => '#1a202c',
                        'created_at' => now(), 'updated_at' => now(),
                    ]
                );
                Log::info('✓ Site Settings seeded');
            }
        } catch (Exception $e) {
            Log::error('❌ Site Settings failed: '.$e->getMessage());
        }
    }

    private function seedContactInfo()
    {
        try {
            ContactInfo::firstOrCreate(
                ['email' => 'info@elhamdimport.online'],
                [
                    'phone' => '+20 2 12345678',
                    'address' => 'القنطرة غرب، الجيزة، مصر', // Matching seed-backup.ts
                    'map_url' => 'https://maps.google.com/?q=30.0131,31.2089',
                    'working_hours' => [
                        'Saturday' => '9:00-17:00',
                        'Sunday' => '9:00-17:00',
                        'Monday' => '9:00-17:00',
                        'Tuesday' => '9:00-17:00',
                        'Wednesday' => '9:00-17:00',
                        'Thursday' => '9:00-17:00',
                        'Friday' => 'مغلق',
                    ],
                ]
            );
            Log::info('✓ Contact Info seeded');
        } catch (Exception $e) {
            Log::error('❌ Contact Info failed: '.$e->getMessage());
        }
    }

    private function seedCompanyStats()
    {
        try {
            $stats = [
                ['label' => 'عملاء سعداء', 'value' => '5000+', 'icon' => 'fas fa-users', 'order' => 1],
                ['label' => 'مركبة مباعة', 'value' => '10000+', 'icon' => 'fas fa-truck', 'order' => 2],
                ['label' => 'سنوات خبرة', 'value' => '14+', 'icon' => 'fas fa-award', 'order' => 3],
                ['label' => 'فروع', 'value' => '2', 'icon' => 'fas fa-map-marker-alt', 'order' => 4],
            ];
            foreach ($stats as $stat) {
                CompanyStat::firstOrCreate(['label' => $stat['label']], $stat);
            }
            Log::info('✓ Company Stats seeded');
        } catch (Exception $e) {
            Log::error('❌ Company Stats failed: '.$e->getMessage());
        }
    }

    private function seedCompanyValues()
    {
        try {
            $values = [
                ['title' => 'الجودة', 'description' => 'نقدم منتجات وخدمات عالية الجودة تلبي أعلى المعايير', 'icon' => 'fas fa-shield-alt', 'order' => 1],
                ['title' => 'الموثوقية', 'description' => 'نضمن موثوقية عالية في جميع منتجاتنا وخدماتنا', 'icon' => 'fas fa-check-circle', 'order' => 2],
                ['title' => 'خدمة العملاء', 'description' => 'نقدم خدمة عملاء ممتازة على مدار الساعة', 'icon' => 'fas fa-headset', 'order' => 3],
                ['title' => 'الابتكار', 'description' => 'نسعى دائماً للابتكار وتطوير حلول جديدة', 'icon' => 'fas fa-lightbulb', 'order' => 4],
            ];
            foreach ($values as $val) {
                CompanyValue::firstOrCreate(['title' => $val['title']], $val);
            }
            Log::info('✓ Company Values seeded');
        } catch (Exception $e) {
            Log::error('❌ Company Values failed: '.$e->getMessage());
        }
    }

    private function seedTimeline()
    {
        try {
            $events = [
                ['title' => 'تأسيس الشركة', 'event_date' => '2010-01-01', 'description' => 'تأسست شركة الحمد لاستيراد السيارات كموزع معتمد لتاتا موتورز في مدن القناة', 'order' => 1],
                ['title' => 'توسع الخدمات', 'event_date' => '2015-01-01', 'description' => 'إضافة خدمات الصيانة وقطع الغيار', 'order' => 2],
                ['title' => 'التحول الرقمي', 'event_date' => '2020-01-01', 'description' => 'إطلاق النظام الإلكتروني لإدارة المبيعات والخدمات', 'order' => 3],
                ['title' => 'التطوير المستمر', 'event_date' => '2024-01-01', 'description' => 'تحديث النظام وتوسيع قاعدة العملاء', 'order' => 4],
            ];
            foreach ($events as $evt) {
                TimelineEvent::firstOrCreate(['title' => $evt['title']], $evt);
            }
            Log::info('✓ Timeline seeded');
        } catch (Exception $e) {
            Log::error('❌ Timeline failed: '.$e->getMessage());
        }
    }

    private function seedServiceItems()
    {
        try {
            $items = [
                ['name' => 'صيانة دورية', 'description' => 'صيانة دورية شاملة للشاحنات والمركبات التجارية', 'icon' => 'fas fa-wrench', 'order' => 1],
                ['name' => 'قطع غيار أصلية', 'description' => 'توفير قطع غيار أصلية من تاتا موتورز', 'icon' => 'fas fa-box', 'order' => 2],
                ['name' => 'خدمة 24 ساعة', 'description' => 'خدمة طوارئ على مدار الساعة طوال أيام الأسبوع', 'icon' => 'fas fa-clock', 'order' => 3],
                ['name' => 'التأجير', 'description' => 'تأجير شاحنات ومركبات تجارية للشركات والأفراد', 'icon' => 'fas fa-truck', 'order' => 4],
            ];
            foreach ($items as $item) {
                ServiceItem::firstOrCreate(['name' => $item['name']], $item);
            }
            Log::info('✓ Service Items seeded');
        } catch (Exception $e) {
            Log::error('❌ Service Items failed: '.$e->getMessage());
        }
    }

    private function seedCompanyFeatures()
    {
        try {
            $features = ['الجودة', 'الموثوقية', 'خدمة العملاء', 'الابتكار'];
            foreach ($features as $idx => $feat) {
                CompanyFeature::firstOrCreate(
                    ['title' => $feat],
                    ['description' => 'ميزة تنافسية لشركتنا', 'order' => $idx + 1]
                );
            }
            Log::info('✓ Company Features seeded');
        } catch (Exception $e) {
            Log::error('❌ Company Features failed: '.$e->getMessage());
        }
    }

    private function seedServiceTypes()
    {
        try {
            $types = [
                ['name' => 'صيانة دورية', 'description' => 'صيانة دورية', 'duration' => 120, 'price' => 500, 'category' => 'MAINTENANCE'],
                ['name' => 'تغيير زيت', 'description' => 'تغيير زيت', 'duration' => 60, 'price' => 200, 'category' => 'MAINTENANCE'],
            ];
            foreach ($types as $type) {
                try {
                    ServiceType::firstOrCreate(
                        ['name' => $type['name']],
                        array_merge($type, ['is_active' => true])
                    );
                } catch (Exception $e) {
                    Log::error("❌ ServiceType {$type['name']} failed: ".$e->getMessage());
                }
            }
            Log::info('✓ Service Types seeded');
        } catch (Exception $e) {
            Log::error('❌ Service Types block failed: '.$e->getMessage());
        }
    }

    private function seedFaqs()
    {
        try {
            $faqs = [
                // About Cars
                ['question' => 'هل السيارات جديدة أم مستعملة؟', 'answer' => 'نحن نستورد سيارات جديدة بالكامل من المصنعين المعتمدين، مع ضمان المصنع الأصلي.', 'category' => 'Vehicles', 'order' => 1],
                ['question' => 'ما هي الدول التي نستورد منها؟', 'answer' => 'نستورد السيارات من اليابان، كوريا، ألمانيا، والولايات المتحدة الأمريكية من وكلاء معتمدين.', 'category' => 'Vehicles', 'order' => 2],
                ['question' => 'هل يمكن طلب سيارة معينة؟', 'answer' => 'نعم، يمكنكم طلب أي سيارة مع المواصفات المطلوبة وسنقوم باستيرادها خصيصاً لكم.', 'category' => 'Vehicles', 'order' => 3],
                ['question' => 'كم تستغرق عملية الاستيراد؟', 'answer' => 'تستغرق عملية الاستيراد من 4 إلى 8 أسابيع حسب بلد المنشأ والإجراءات الجمركية.', 'category' => 'Vehicles', 'order' => 4],

                // Warranty & Maintenance
                ['question' => 'ما هو الضمان المقدم على السيارات؟', 'answer' => 'نقدم ضماناً شاملاً لمدة سنة أو 20,000 كم، مع إمكانية التمديد لفترات أطول.', 'category' => 'Maintenance', 'order' => 5],
                ['question' => 'أين يمكنني صيانة السيارة؟', 'answer' => 'لدينا مراكز صيانة معتمدة في جميع المحافظات، بالإضافة إلى الوكلاء المعتمدين من الشركة المصنعة.', 'category' => 'Maintenance', 'order' => 6],
                ['question' => 'هل قطع الغيار أصلية؟', 'answer' => 'نعم، نستخدم فقط قطع الغيار الأصلية من الشركة المصنعة لضمان الجودة والأداء.', 'category' => 'Maintenance', 'order' => 7],
                ['question' => 'ماذا يغطي الضمان؟', 'answer' => 'يغطي الضمان جميع الأعطال الميكانيكية والكهربائية، مع استثناء التآكل الطبيعي والأضرار الناتجة عن الحوادث.', 'category' => 'Maintenance', 'order' => 8],

                // Finance
                ['question' => 'هل تقدمون تمويلاً للسيارات؟', 'answer' => 'نعم، نتعاون مع أكبر البنوك وشركات التمويل لتقديم أفضل شروط التمويل.', 'category' => 'Finance', 'order' => 9],
                ['question' => 'ما هي المبالغ المقدمة المطلوبة؟', 'answer' => 'تتراوح المبالغ المقدمة من 20% إلى 30% حسب نوع السيارة وشروط التمويل.', 'category' => 'Finance', 'order' => 10],
                ['question' => 'ما هي طرق الدفع المتاحة؟', 'answer' => 'نقبل الدفع النقدي، الشيكات، البطاقات الائتمانية، والتحويل البنكي.', 'category' => 'Finance', 'order' => 11],
                ['question' => 'هل يمكن التقسيط بدون مقدم؟', 'answer' => 'بعض السيارات المتوسطة تتوفر فيها خطط تقسيط بدون مقدم للعملاء ذوي السجل الائتماني الجيد.', 'category' => 'Finance', 'order' => 12],

                // Procedures
                ['question' => 'ما هي المستندات المطلوبة للشراء؟', 'answer' => 'بطاقة الرقم القومي سارية، إثبات العنوان، وإيصال مرتب حديث للموظفين.', 'category' => 'Procedures', 'order' => 13],
                ['question' => 'هل تتعاملون مع الشركات؟', 'answer' => 'نعم، لدينا قسم خاص للتعامل مع الشركات وتوفير أساطيل السيارات.', 'category' => 'Procedures', 'order' => 14],
                ['question' => 'هل السيارات مؤمنة؟', 'answer' => 'نوفر تأميناً شاملاً على جميع السيارات مع أفضل شركات التأمين في مصر.', 'category' => 'Procedures', 'order' => 15],
                ['question' => 'ما هي رسوم التسجيل؟', 'answer' => 'رسوم التسجيل والجمارك مضمنة في السعر النهائي للسيارة.', 'category' => 'Procedures', 'order' => 16],
            ];

            foreach ($faqs as $faq) {
                \App\Models\Faq::firstOrCreate(['question' => $faq['question']], $faq);
            }
            Log::info('✓ FAQs seeded');
        } catch (Exception $e) {
            Log::error('❌ FAQs failed: '.$e->getMessage());
        }
    }

    private function seedCompanyInfo()
    {
        try {
            \App\Models\CompanyInfo::firstOrCreate(
                ['title' => 'مرحباً بك في الحمد للسيارات'],
                [
                    'subtitle' => 'الموزع المعتمد لسيارات تاتا في مدن القناة',
                    'description' => 'نحن نفخر بكوننا الوجهة الأولى لسيارات تاتا في بورسعيد والإسماعيلية. نقدم أحدث الموديلات مع خدمات متكاملة تشمل البيع، الصيانة، وقطع الغيار الأصلية.',
                    'image_url' => 'images/showroom-luxury.jpg', // Asset helper used in view
                    'features' => [
                        'أحدث موديلات تاتا 2024',
                        'ضمان المصنع لمدة 3 سنوات',
                        'ضمان صيانة على مدار الساعة',
                        'تمويل سيارات بأفضل الأسعار',
                    ],
                    'cta_buttons' => [
                        ['text' => 'استعرض السيارات', 'link' => route('vehicles.index'), 'variant' => 'primary'],
                        ['text' => 'قيادة تجريبية', 'link' => route('contact'), 'variant' => 'secondary'],
                    ],
                ]
            );
            Log::info('✓ CompanyInfo seeded');
        } catch (Exception $e) {
            Log::error('❌ CompanyInfo failed: '.$e->getMessage());
        }
    }

    private function seedPermissionsAndRoles()
    {
        try {
            $permissions = ['vehicles.view', 'vehicles.create'];
            foreach ($permissions as $permName) {
                Permission::firstOrCreate(['name' => $permName], ['guard_name' => 'web']);
            }
            $roles = ['Super Admin', 'Admin', 'Customer'];
            foreach ($roles as $roleName) {
                RoleTemplate::firstOrCreate(['name' => $roleName], ['label' => $roleName]);
            }
            Log::info('✓ Permissions & Roles seeded');
        } catch (Exception $e) {
            Log::error('❌ Permissions & Roles failed: '.$e->getMessage());
        }
    }

    private function seedBranches()
    {
        return Branch::firstOrCreate(
            ['code' => 'ELHAMD-MAIN'],
            [
                'name' => 'الفرع الرئيسي',
                'address' => 'القنطرة غرب',
                'phone' => '+20 12345678',
                'email' => 'info@elhamdimport.online',
                'is_active' => true,
                'opening_date' => '2010-01-01',
                'currency' => 'EGP',
                'timezone' => 'Africa/Cairo',
                'settings' => json_encode(['services' => []]),
            ]
        );
    }

    private function seedUsers($branch)
    {
        try {
            $seedPassword = app()->environment(['local', 'testing'])
                ? env('SEED_DEFAULT_PASSWORD', 'password')
                : Str::random(32);

            User::firstOrCreate(
                ['email' => 'admin@elhamdimport.online'],
                [
                    'name' => 'مدير النظام',
                    'password' => Hash::make($seedPassword),
                    'branch_id' => $branch->id,
                ]
            );
            Log::info('✓ Users seeded');
        } catch (Exception $e) {
            Log::error('❌ Users failed: '.$e->getMessage());
        }
    }

    private function seedVehicles($branch)
    {
        Log::info('Entering seedVehicles');

        $commercial = [
            [
                'make' => 'Tata', 'model' => 'PRIMA 3328.K', 'year' => 2024, 'price' => 1500000, 'stock_number' => 'TAT-001',
                'description' => 'شاحنة تاتا بريما 3328.K - قوة وأداء استثنائي للنقل الثقيل',
                'images' => ['/uploads/vehicles/PRIMA-3328.K-1.jpg', '/uploads/vehicles/prima-3328k-1.jpg', '/uploads/vehicles/prima-3328k-2.jpg'],
                'category' => 'TRUCK', 'fuel_type' => 'DIESEL', 'featured' => true,
            ],
            [
                'make' => 'Tata', 'model' => 'LP 613', 'year' => 2024, 'price' => 850000, 'stock_number' => 'TAT-002',
                'description' => 'تاتا LP 613 - شاحنة نقل متوسطة متعددة الاستخدامات',
                'images' => ['/uploads/vehicles/LP-613-1.jpg', '/uploads/vehicles/lp-613-2.jpg'],
                'category' => 'VAN', 'fuel_type' => 'DIESEL', 'featured' => true,
            ],
            [
                'make' => 'Tata', 'model' => 'LPT 1618', 'year' => 2024, 'price' => 650000, 'stock_number' => 'TAT-003',
                'description' => 'تاتا LPT 1618 - شاحنة نقل ثقيلة بمحرك ديزل قوي',
                'images' => ['/uploads/vehicles/LPT-1618-1.jpg', '/uploads/vehicles/TAT-003-1.jpg'],
                'category' => 'COMMERCIAL', 'fuel_type' => 'DIESEL', 'featured' => true,
            ],
            [
                'make' => 'Tata', 'model' => 'ULTRA T.7', 'year' => 2024, 'price' => 450000, 'stock_number' => 'TAT-004',
                'description' => 'تاتا ULTRA T.7 - ربع نقل خفيف وعملي للمدن والطرق السريعة',
                'images' => ['/uploads/vehicles/ULTRA-T.7-1.jpg', '/uploads/vehicles/ultra-t7-1.jpg'],
                'category' => 'COMMERCIAL', 'fuel_type' => 'DIESEL', 'featured' => true,
            ],
            [
                'make' => 'Tata', 'model' => 'XENON SC', 'year' => 2024, 'price' => 380000, 'stock_number' => 'TAT-005',
                'description' => 'تاتا زينون SC - بيك أب قوي ومتين للأعمال الشاقة',
                'images' => ['/uploads/vehicles/XENON-SC-1.jpg', '/uploads/vehicles/xenon-sc-2.jpg'],
                'category' => 'TRUCK', 'fuel_type' => 'DIESEL', 'featured' => true,
            ],
            [
                'make' => 'Tata', 'model' => 'LPT 613 Tipper', 'year' => 2024, 'price' => 420000, 'stock_number' => 'TAT-006',
                'description' => 'تاتا LPT 613 قلاب - مثالية لأعمال البناء والنقل',
                'images' => ['/uploads/vehicles/LPT-613-1.jpg', '/uploads/vehicles/lpt613-tipper-1.jpg'],
                'category' => 'COMMERCIAL', 'fuel_type' => 'DIESEL', 'featured' => true,
            ],
            [
                'make' => 'Tata', 'model' => 'ULTRA T.9', 'year' => 2024, 'price' => 520000, 'stock_number' => 'TAT-007',
                'description' => 'تاتا ULTRA T.9 - نصف نقل بقدرة تحمل عالية وأداء متميز',
                'images' => ['/uploads/vehicles/ULTRA-T.9-1.jpg', '/uploads/vehicles/ultra-t9-1.jpg'],
                'category' => 'COMMERCIAL', 'fuel_type' => 'DIESEL', 'featured' => true,
            ],
        ];

        Log::info('Processing Vehicles Loop with '.count($commercial).' items.');

        foreach ($commercial as $v) {
            $stockNumber = $v['stock_number'];
            Log::info("Checking $stockNumber");

            try {
                $existing = Vehicle::where('stock_number', $stockNumber)->first();

                if (! $existing) {
                    Log::info("Creating $stockNumber");
                    $data = [
                        'make' => $v['make'],
                        'model' => $v['model'],
                        'year' => $v['year'],
                        'price' => $v['price'],
                        'stock_number' => $stockNumber,
                        'stock_quantity' => 1,
                        'vin' => $v['vin'] ?? Str::upper(Str::random(17)),
                        'description' => $v['description'] ?? 'Vehicle Description',
                        'category' => $v['category'] ?? 'COMMERCIAL',
                        'fuel_type' => $v['fuel_type'] ?? 'DIESEL',
                        'transmission' => $v['transmission'] ?? 'MANUAL',
                        'status' => 'AVAILABLE',
                        'featured' => $v['featured'] ?? true,
                        'branch_id' => $branch->id,
                    ];

                    if (Schema::hasColumn('vehicles', 'features') && isset($v['features'])) {
                        $data['features'] = $v['features'];
                    }

                    $vehicle = Vehicle::create($data);

                    if (isset($v['images'])) {
                        foreach ($v['images'] as $idx => $path) {
                            try {
                                VehicleImage::create([
                                    'vehicle_id' => $vehicle->id,
                                    'image_url' => $path,
                                    'is_primary' => $idx === 0,
                                    'order' => $idx,
                                ]);
                            } catch (Exception $e) {
                                Log::error("❌ Image failed for $stockNumber: ".$e->getMessage());
                            }
                        }
                    }
                } else {
                    Log::info("Skipping existing: $stockNumber");
                }
            } catch (Exception $e) {
                Log::error("❌ Vehicle $stockNumber failed: ".$e->getMessage());
            }
        }
    }

    private function seedSliders()
    {
        try {
            $sliders = [
                [
                    'title' => 'الحمد للسيارات التجارية',
                    'subtitle' => 'الموزع المعتمد لسيارات تاتا موتورز في مدن القناة',
                    'image_url' => '/uploads/sliders/slider-slider-0.jpeg',
                    'cta_text' => 'استعرض المركبات',
                    'cta_link' => '/vehicles',
                    'order' => 1,
                    'is_active' => true,
                ],
                [
                    'title' => 'شاحنات تاتا PRIMA',
                    'subtitle' => 'القوة والموثوقية في النقل الثقيل',
                    'image_url' => '/uploads/sliders/slider-slider-1.jpeg',
                    'cta_text' => 'اكتشف المزيد',
                    'cta_link' => '/vehicles',
                    'order' => 2,
                    'is_active' => true,
                ],
                [
                    'title' => 'خدمات متكاملة',
                    'subtitle' => 'صيانة وقطع غيار أصلية على مدار الساعة',
                    'image_url' => '/uploads/sliders/slider-slider-2.jpeg',
                    'cta_text' => 'احجز موعد صيانة',
                    'cta_link' => '/service-booking',
                    'order' => 3,
                    'is_active' => true,
                ],
            ];
            foreach ($sliders as $slide) {
                Slider::firstOrCreate(['title' => $slide['title']], $slide);
            }
            Log::info('✓ Sliders seeded');
        } catch (Exception $e) {
            Log::error('❌ Sliders failed: '.$e->getMessage());
        }
    }
}
