<?php

use Illuminate\Foundation\Inspiring;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Schedule;

// Scheduled Commands
Schedule::command('invoices:mark-overdue')->dailyAt('01:00');
Schedule::command('bookings:send-reminders')->dailyAt('09:00');
Schedule::command('cleanup:temp')->weeklyOn(0, '03:00');

Artisan::command('inspire', function () {
    $this->comment(Inspiring::quote());
})->purpose('Display an inspiring quote');

Artisan::command('verify:system', function () {
    $this->info('Starting System Verification...');

    try {
        DB::beginTransaction();

        // 1. Create Customer
        $this->info('1. Creating Customer...');
        $customer = \App\Models\User::create([
            'name' => 'Verify Customer',
            'email' => 'verify.'.time().'@example.com',
            'password' => \Illuminate\Support\Facades\Hash::make('password'),
            'role' => 'customer',
            'phone' => '1234567890',
        ]);
        $this->info('   OK (ID: '.$customer->id.')');

        // 2. Create Service Type (Required for Booking)
        $this->info('2. Creating Service Type...');
        $serviceType = \App\Models\ServiceType::create([
            'name' => 'Verification Service',
            'description' => 'Service for verification',
            'duration' => 60,
            'price' => 500,
            'category' => 'General Maintenance', // Verify if this is Enum or String
            'is_active' => true,
        ]);
        $this->info('   OK (ID: '.$serviceType->id.')');

        // 3. Create Vehicle
        $this->info('3. Creating Vehicle...');
        $vehicle = \App\Models\Vehicle::create([
            'make' => 'Honda',
            'model' => 'Civic',
            'year' => 2024,
            'vin' => 'VIN-VERIFY-'.time(),
            'stock_number' => 'STK-VERIFY-'.time(),
            'status' => 'AVAILABLE',
            'price' => 30000,
            'stock_quantity' => 1,
            'category' => 'Sedan',
            'fuel_type' => 'Hybrid',
            'transmission' => 'Automatic',
        ]);
        $this->info('   OK (ID: '.$vehicle->id.')');

        // 4. Create Booking
        $this->info('4. Creating Booking...');
        $booking = \App\Models\ServiceBooking::create([
            'customer_id' => $customer->id,
            'vehicle_id' => $vehicle->id,
            'service_type_id' => $serviceType->id, // Use ID
            'date' => now()->addDays(2),
            'time_slot' => '02:00 PM',
            'status' => 'PENDING',
        ]);
        $this->info('   OK (ID: '.$booking->id.')');

        // 5. Create Invoice
        $this->info('5. Creating Invoice...');
        $invoice = \App\Models\Invoice::create([
            'invoice_number' => 'INV-VERIFY-'.time(),
            'customer_id' => $customer->id,
            'vehicle_id' => $vehicle->id,
            'type' => 'SERVICE',
            'status' => 'DRAFT',
            'payment_status' => 'PENDING',
            'issue_date' => now(),
            'due_date' => now()->addDays(7),
            'subtotal' => 500,
            'tax_amount' => 70,
            'total_amount' => 570,
            'paid_amount' => 0,
            'created_by' => null, // UUID string or null
        ]);
        $this->info('   OK (ID: '.$invoice->id.')');

        // 6. Record Payment
        $this->info('6. Recording Payment...');
        $payment = \App\Models\Payment::create([
            'customer_id' => $customer->id,
            'invoice_id' => $invoice->id,
            'amount' => 570,
            'payment_method' => 'CASH',
            'transaction_id' => 'TXN-VERIFY-'.time(),
            'status' => 'COMPLETED',
            'currency' => 'EGP',
        ]);

        // Update Invoice Logic Check
        $invoice->refresh();
        $invoice->paid_amount += 570;
        if ($invoice->paid_amount >= $invoice->total_amount) {
            $invoice->status = 'PAID';
            $invoice->payment_status = 'PAID';
        }
        $invoice->save();
        $this->info('   OK (ID: '.$payment->id.', Invoice Status: '.$invoice->status.')');

        // 7. Support Ticket
        $this->info('7. Creating Ticket...');
        $ticket = \App\Models\Ticket::create([
            'ticket_number' => 'TCK-VERIFY-'.time(),
            'customer_id' => $customer->id,
            'subject' => 'Verify Logic',
            'description' => 'Test description',
            'status' => 'OPEN',
            'priority' => 'HIGH',
            'category' => 'Technical',
        ]);
        $this->info('   OK (ID: '.$ticket->id.')');

        DB::commit();
        $this->info('All Verification Steps Passed Successfully!');

    } catch (\Exception $e) {
        DB::rollBack();
        file_put_contents(base_path('verification_error.txt'), $e->getMessage()."\n".$e->getTraceAsString());
        $this->error('Verification Failed: '.$e->getMessage());
        exit(1);
    }
});
